<?php

defined( 'ABSPATH' ) or die( 'Access denied' );

if( ! class_exists( 'A_W_F_premium_admin' ) ) {

  class A_W_F_premium_admin extends A_W_F_admin {

    public function __construct() {
      parent::__construct();
      
      if( empty( get_option( 'awf_premium' ) ) ) { add_action( 'wp_loaded', array( $this, 'upgrade_to_premium' ) ); }

      add_action( 'init', array( $this, 'admin_get_query_actions' ), 5 );
      add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_premium_admin_scripts') );
      add_action( 'admin_menu', array( $this, 'add_premium_admin_menu_options' ) );
      
      add_filter( 'awf_range_slider_style_options', array( $this, 'add_range_slider_style_options' ) );
      add_filter( 'awf_color_filter_style_options', array( $this, 'add_color_filter_style_options' ) );

      $this->filter_types['single']['styles'][] = 'images';
      $this->filter_types['single']['styles'][] = 'custom-terms';
      $this->filter_types['multi']['styles'][] = 'images';
      $this->filter_types['multi']['styles'][] = 'custom-terms';
      $this->filter_types['range']['styles'][] = 'images';
      $this->filter_types['range']['styles'][] = 'custom-terms';
      $this->filter_style_limitations['taxonomy']['range'] = array( 'range-slider', 'radios', 'icons', 'labels', 'images', 'custom-terms' );
      $this->filter_style_limitations['rating']['range'][] = 'range-slider';
      $this->filter_style_limitations['rating']['range'][] = 'range-stars';
    }

    public function enqueue_premium_admin_scripts( $hook ) {
      if( isset( $_GET['tab'] ) && $_GET['tab'] === 'annasta-filters' ) {
        
        if( isset( $_GET['awf-preset'] ) ) {
          wp_enqueue_media();
          wp_enqueue_script( 'awf-media-handler', A_W_F_PLUGIN_URL . '/code/js/awf-admin-media-handler.js', array( 'awf-admin' ), A_W_F::$plugin_version );
        }

        wp_enqueue_script( 'awf-premium-admin', A_W_F_PLUGIN_URL . '/code/js/awf-premium-admin.js', array( 'awf-admin' ), A_W_F::$plugin_version );
      }
    }

    public function add_premium_admin_menu_options() {
      add_submenu_page( 'annasta-filters', '', __( 'Templates', 'annasta-filters' ), 'manage_woocommerce', 'annasta-filters-templates-settings', array( $this, 'safe_redirect_to_templates_settings' ), 3 );
    }
    
    public function safe_redirect_to_templates_settings() {
      $redirect_url = admin_url( 'admin.php?page=wc-settings&tab=annasta-filters&section=templates-settings' );

      wp_safe_redirect( $redirect_url );
      exit();
    }

    public function upgrade_to_premium() {
      
      foreach( A_W_F::$presets as $preset_id => $preset ) {
        foreach( $preset['filters'] as $filter_id => $position ) {
          $filter = new A_W_F_filter( $preset_id, $filter_id );
          
          $filter_data = array( 'module' => $filter->module, 'title' => '' );
          if( 'taxonomy' === $filter->module ) {
            $filter_data['taxonomy'] = get_taxonomy( $filter->settings['taxonomy'] );
            if( false === $filter_data['taxonomy'] ) { continue; }
          }
          
          $premium_settings = $this->get_module_defaults( $filter_data );
          
          foreach( $premium_settings as $setting => $value ) {
            if( isset( $filter->settings[$setting] ) && ( ! is_null( $filter->settings[$setting] ) ) ) {
              $premium_settings[$setting] = $filter->settings[$setting];
            }
          }
          
          update_option( $filter->prefix. 'settings', $premium_settings );
        }
      }
      
      update_option( 'awf_premium', true );
    }

    public function admin_get_query_actions() {
      if( isset( $_GET['awf-action'] ) ) {
        if( 'create-preset' === $_GET['awf-action'] ) {
          $this->check_permissions();
          $id = $this->create_preset();
          $this->redirect_to_presets_tab( array( 'awf-preset' => $id ) );

        } elseif( 'delete-preset' === $_GET['awf-action'] ) {
          $this->delete_preset();
        }
      }
    }
    
    /**
     * Process premium admin AJAX requests.
     *
     * @see A_W_F_admin::ajax_controller()
     *
     * @param mixed $filter A_W_F_filter instance or false if there is no filter provided by the call.
     *
     * @return mixed
     */

    protected function premium_ajax_controller( $filter = false ) {
      if( $filter ) {
        if( 'update-range-positions' === $_POST['awf_action'] ) {
          $this->update_range_positions( $filter );

        } elseif( 'popup-filter-templates' === $_POST['awf_action'] ) {
          $this->display_apply_filter_template_popup( $filter );

        } elseif( 'apply-filter-template' === $_POST['awf_action'] ) {
          $templates = $this->get_templates();
          $template_id = intval( $_POST['awf_filter_template_id'] );

          if( isset( $templates['filters'][$template_id] ) ) {
            $this->apply_filter_template( $filter, $templates['filters'][$template_id] );
          }

        } elseif( 'apply-filter-clone' === $_POST['awf_action'] ) {
          $this->apply_filter_clone( $filter );

        } elseif( 'create-filter-template' === $_POST['awf_action'] ) {
          $this->create_filter_template( $filter, sanitize_text_field( $_POST['awf_new_filter_template_name'] ) );
          echo json_encode( array( 'filter_name' => $filter->name ) );
        }

      } else {
          
        if( 'popup-custom-layout' === $_POST['awf_action'] ) {
          $this->display_custom_layout_popup();

        } elseif( 'add-custom-layout-column' === $_POST['awf_action'] ) {
          $this->add_custom_layout_column();

        } elseif( 'delete-custom-layout-column' === $_POST['awf_action'] ) {
          $this->delete_custom_layout_column();

        } elseif( 'update-custom-layout' === $_POST['awf_action'] ) {
          $this->update_custom_layout();

        } elseif( 'popup-preset-templates' === $_POST['awf_action'] ) {
          $this->display_apply_preset_template_popup( intval( $_POST['awf_preset_id'] ) );

        } elseif( 'apply-preset-template' === $_POST['awf_action'] ) {
          $this->apply_preset_template( intval( $_POST['awf_preset_id'] ), intval( $_POST['awf_preset_template_id'] ) );

        } elseif( 'apply-preset-clone' === $_POST['awf_action'] ) {
          $this->apply_preset_clone( intval( $_POST['awf_preset_id'] ), intval( $_POST['awf_clone_preset_id'] ) );

        } elseif( 'popup-create-preset-template' === $_POST['awf_action'] ) {
          $this->display_create_preset_template_popup();

        } elseif( 'create-preset-template' === $_POST['awf_action'] ) {
          $this->create_preset_template( intval( $_POST['awf_preset_id'] ), sanitize_text_field( wp_unslash( $_POST['awf_new_preset_template_name'] ) ) );

        } elseif( 'popup-rename-preset-template' === $_POST['awf_action'] ) {
          $this->display_rename_preset_template_popup( intval( $_POST['awf_preset_template_id'] ) );

        } elseif( 'rename-preset-template' === $_POST['awf_action'] ) {
          $this->rename_preset_template( intval( $_POST['awf_preset_template_id'] ), sanitize_text_field( wp_unslash( $_POST['awf_preset_template_name'] ) ) );

        } elseif( 'export-preset-template' === $_POST['awf_action'] ) {
          $this->export_preset_template( intval( $_POST['awf_preset_template_id'] ) );

        } elseif( 'delete-preset-template' === $_POST['awf_action'] ) {
          $this->delete_preset_template( intval( $_POST['awf_preset_template_id'] ) );

        } elseif( 'popup-create-filter-template' === $_POST['awf_action'] ) {
          $this->display_create_filter_template_popup();
          
        } elseif( 'popup-create-filter-template-change-preset' === $_POST['awf_action'] ) {
          $this->display_create_filter_template_popup( intval( $_POST['awf_preset_id'] ) );

        } elseif( 'import-template' === $_POST['awf_action'] ) {
          if( 'preset' === $_POST['awf_json_template_type'] ) {
            echo( $this->import_preset_template( $_POST['awf_json_template'] ) );
          } elseif( 'filter' === $_POST['awf_json_template_type'] ) {
            $this->import_filter_template( $_POST['awf_json_template'] );
          }
        } elseif( 'popup-rename-filter-template' === $_POST['awf_action'] ) {
          $this->display_rename_filter_template_popup( intval( $_POST['awf_filter_template_id'] ) );

        } elseif( 'rename-filter-template' === $_POST['awf_action'] ) {
          $this->rename_filter_template( intval( $_POST['awf_filter_template_id'] ), sanitize_text_field( wp_unslash( $_POST['awf_filter_template_name'] ) ) );

        } elseif( 'export-filter-template' === $_POST['awf_action'] ) {
          $this->export_filter_template( intval( $_POST['awf_filter_template_id'] ) );

        } elseif( 'delete-filter-template' === $_POST['awf_action'] ) {
          $this->delete_filter_template( intval( $_POST['awf_filter_template_id'] ) );

        } elseif( 'clone-preset' === $_POST['awf_action'] ) {
          $this->clone_preset( intval( $_POST['awf_preset_id'] ) );

        }
      }
    }
    
    private function create_preset() {
      $new_preset_id = empty( A_W_F::$presets ) ? 1 : max( array_keys( A_W_F::$presets ) ) + 1;
      $position = count( A_W_F::$presets ) + 1;
      
      A_W_F::$presets[$new_preset_id] = array(
        'position' => $position,
        'filters' => array(),
        'associations' => array( 0 => 'all' ),
        'overrides' => array()
      );

      uasort( A_W_F::$presets, function( $a, $b ) {
          return $a['position'] - $b['position'];
      });

      update_option( 'awf_presets', A_W_F::$presets );
      
      $new_preset = new A_W_F_preset( $new_preset_id );
      $options = $new_preset->get_preset_options();
      
      foreach( $options as $option => $value ) {
        update_option( 'awf_preset_' . $new_preset_id . '_' . $option, $value );
      }

      return $new_preset_id;
    }

    private function clone_preset( $parent_id ) {

      $template_id = $this->create_preset_template( $parent_id, 'Preset #' . $parent_id . ' Clone' );

      if( false === $template_id ) { die(); }
      
      $new_preset_id = $this->create_preset();

      $this->apply_preset_template( $new_preset_id, $template_id );
      update_option( 'awf_preset_' . $new_preset_id . '_name', sprintf( __( '%1$s [CLONE]', 'annasta-filters' ), get_option( 'awf_preset_' . $parent_id . '_name', '' ) ) );
      $this->delete_preset_template( $template_id );

      echo admin_url( 'admin.php?page=wc-settings&tab=annasta-filters&awf-preset=' . $new_preset_id );
    }

    private function delete_preset() {
      $this->check_permissions();
      
      $delete_id = (int) $_GET['awf-delete-preset'];

      if( isset( A_W_F::$presets[$delete_id] ) ) {

        foreach( A_W_F::$presets[$delete_id]['filters'] as $filter_id => $filter_position ) {
          $filter = new A_W_F_filter( $delete_id, $filter_id );
          $this->delete_filter( $filter );
        }

        A_W_F::build_query_vars();
        $this->clear_product_counts_cache();

        $deleted_position = A_W_F::$presets[$delete_id]['position'];
        unset( A_W_F::$presets[$delete_id] );

        if( ! empty( A_W_F::$presets ) ) {
          foreach( A_W_F::$presets as $preset_id => $preset ) {
            if( $preset['position'] > $deleted_position ) {
              A_W_F::$presets[$preset_id]['position'] -= 1;
            }
          }
        }

        update_option( 'awf_presets', A_W_F::$presets );
      }
      
      $delete_preset = new A_W_F_preset( $delete_id );
      $options = $delete_preset->get_preset_options();
      foreach( $options as $option => $value ) {
        delete_option( 'awf_preset_' . $delete_id . '_' . $option );
      }

      $this->redirect_to_presets_tab();
    }
    
    public function awf_display_preset_btns( $preset_id = 0, $admin_url = '' ) {
      ?>
        <a class="button button-secondary awf-fa-icon awf-fas-icon awf-fa-delete-btn" href="<?php echo esc_url( add_query_arg( array( 'awf-action' => 'delete-preset', 'awf-nonce' => wp_create_nonce( 'awf-delete-preset' ), 'awf-delete-preset' => $preset_id ), $admin_url ) ); ?>" title="<?php esc_attr_e( 'Delete preset', 'annasta-filters' ); ?>"></a>
        <button class="button button-secondary awf-fa-icon awf-fa-clone-btn" title="<?php esc_attr_e( 'Clone preset', 'annasta-filters' ); ?>" data-preset-id="<?php esc_attr_e( $preset_id ); ?>"></button>
      <?php
    }
    
    public function display_presets_list_title_buttons( $admin_url = '' ) {
      ?>
				<a class="button button-secondary awf-fa-icon-text-btn awf-fa-add-btn awf-create-preset-btn" href="<?php echo esc_url( add_query_arg( array( 'awf-action' => 'create-preset', 'awf-nonce' => wp_create_nonce( 'awf-create-preset' ) ),  $admin_url ) ); ?>"><?php esc_html_e( 'Add preset', 'annasta-filters' ); ?></a>
      <?php
    }
    
    public function awf_display_presets_list_footer( $admin_url = '' ) {
      ?>
				<a class="button button-secondary awf-fa-icon-text-btn awf-fa-add-btn awf-create-preset-btn" href="<?php echo esc_url( add_query_arg( array( 'awf-action' => 'create-preset', 'awf-nonce' => wp_create_nonce( 'awf-create-preset' ) ),  $admin_url ) ); ?>"><?php esc_html_e( 'Add preset', 'annasta-filters' ); ?></a>
      <?php
    }
    
    protected function update_presets_positions() {
      $positions = ( isset( $_POST['awf_presets_positions'] ) && is_array( $_POST['awf_presets_positions'] ) ) ?  array_map( 'intval', $_POST['awf_presets_positions'] ) : array();

      $new_positions = array_flip( $positions );

      $check_ids = array_diff_key( $new_positions, A_W_F::$presets );
      
      if( count( $check_ids ) === 0 ) {

        foreach( $new_positions as $preset_id => $position ) {
          $position = absint( sanitize_key( $position ) ) + 1;
          A_W_F::$presets[intval( $preset_id )]['position'] = $position;
        }

        uasort( A_W_F::$presets, function( $a, $b ) {
          return $a['position'] - $b['position'];
        });

        update_option( 'awf_presets', A_W_F::$presets );

      } else {
        wp_send_json_error( array( 'awf_error_message' => __( 'An error occured when updating presets positions.', 'annasta-filters' ) ), 400 );
      }
    }
    
    public function get_preset_settings( $preset ) {
      			
      $settings = parent::get_preset_settings( $preset );

      $settings[190]['options'] = array(
        '1-column'        => __( '1 column', 'annasta-filters' ),
        '2-column'        => __( '2 columns', 'annasta-filters' ),
        '3-column'        => __( '3 columns', 'annasta-filters' ),
        '4-column-flex'   => __( '4 columns', 'annasta-filters' ),
        '4-column'        => __( '4 columns with wide dropdowns', 'annasta-filters' ),
        'custom'          => __( 'Custom', 'annasta-filters' )
      );

      ksort( $settings );

			return $settings;
    }

    private function display_custom_layout_popup() {
      $preset_id = intval( $_POST['awf_preset_id'] );

      $html = '<div class="awf-overlay-popup awf-custom-layout-popup" data-preset-id="' . $preset_id . '">' .
      '<div title="' . esc_attr__( 'Close', 'annasta-filters' ) . '" class="awf-fa-icon awf-close-overlay-popup-btn"></div>' .

      '<div class="awf-overlay-popup-row awf-overlay-popup-header-row">' .
      esc_html__( 'Custom preset layout', 'annasta-filters' ) .
      '</div>' .

      '<div class="awf-overlay-popup-row awf-info-notice">' .
      esc_html__( 'To make the columns spread to the width of their parent container, use the percentage units making sure that the sum of all column widths and spaces between them equals 100%.', 'annasta-filters' ) .
      '</div>'
      ;

      if( ! empty( $preset_id ) ) {

        $customizer_options = get_option( 'awf_customizer_options', array() );

        if( empty( $customizer_options[ 'preset_' . $preset_id . '_custom_layout'] ) ) {
          $settings = array( '', '', '' );
        } else {
          $settings = explode( '|||', $customizer_options[ 'preset_' . $preset_id . '_custom_layout'] );
          if( 3 !== count( $settings ) ) { $settings = array( '', '', '' ); }
        }

        $columns = explode( '||', $settings[0] );
        $columns = array_values( array_filter( $columns ) );
        
        $html .= '<div class="awf-overlay-popup-row awf-custom-layout-columns">';

        foreach( $columns as $i => $cw ) {
          $data = explode( '|', $cw );

          if( 2 !== count( $data ) ) {
            $customizer_options[ 'preset_' . $preset_id . '_custom_layout'] = '';
            update_option( 'awf_customizer_options', $customizer_options );
            break;
          }

          $aw = '';
          $ci = $i + 1;

          if( 'auto' === $data[0] ) {
            $w = '';
            $aw = ' checked';
          } else {
            $w = abs( floatval( $data[0] ) );
          }

          $html .= '<div class="awf-custom-layout-column-container' . ( empty( $aw ) ? '' : ' awf-autowidth-on' ) . '" data-column-n="' . $ci . '">';
          $html .= '<div class="awf-delete-custom-layout-column-btn" data-column-n="' . $ci . '" title="' . esc_html__( 'Delete', 'annasta-filters') . '"><i class="fas fa-times"></i></div>';
          $html .= '<div class="awf-custom-layout-input-label">' . sprintf( esc_html__( 'Column #%1$s', 'annasta-filters' ), esc_html( $ci ) ) . '</div>';
          $html .= '<div class="awf-custom-layout-width-container">';
          $html .= '<input type="number" class="awf-custom-layout-width" id="awf-custom-layout-width-' . $ci . '" value="' . $w . '" />';
          $html .= A_W_F::$admin->build_select_html( array(
            'id' => 'awf-custom-layout-units-' . $ci, 
            'class' => 'awf-custom-layout-units-select', 
            'options' => array( '%' => '%', 'px' => 'px', 'em' => 'em' ), 
            'selected' => empty( $data[1] ) ? '%' : sanitize_text_field( $data[1] )
          ) );
          $html .= '</div>';
          $html .= '<div class="awf-custom-layout-autowidth-container"><input type="checkbox" id="awf-custom-layout-autowidth-' . $ci . '" value="1"' . $aw . ' /><label for="awf-custom-layout-autowidth-' . $ci . '">' . esc_html__( 'Auto width', 'annasta-filters' ) . '</label></div>';
          $html .= '</div>';
        }

        $html .= '<button type="button" title="' . esc_attr__( 'Add column', 'annasta-filters' ) . '" class="button button-secondary awf-add-custom-layout-column-btn" id="awf-add-custom-layout-column-btn">' . esc_html__( 'Add column', 'annasta-filters' ) . '</button>' .

        '</div>';

        $gap = explode( '|', $settings[1] );
        $wrap = $settings[2];

        $html .= '<div class="awf-overlay-popup-row awf-custom-layout-options">' .

        '<div class="awf-custom-layout-gap-container"><div class="awf-custom-layout-input-label">' . esc_html__( 'Space between columns', 'annasta-filters' ) . '</div>' .
        '<input type="number" id="awf-custom-layout-gap" value="' . ( isset( $gap[0] ) ? abs( floatval( $gap[0] ) ) : '' ) . '" />' .
        A_W_F::$admin->build_select_html( array(
          'id' => 'awf-custom-layout-gap-units', 
          'options' => array( '%' => '%', 'px' => 'px', 'em' => 'em' ), 
          'selected' => empty( $gap[1] ) ? '%' : sanitize_text_field( $gap[1] )
        ) ) .
        '</div>' .

        '<div class="awf-custom-layout-wrap-container">' .
        '<input type="checkbox" id="awf-custom-layout-wrap" value="1"' . ( empty( $wrap ) ? '' : ' checked' ) . ' />' .
        '<label for="awf-custom-layout-wrap" class="awf-custom-layout-input-label">' . esc_html__( 'Force one-row display', 'annasta-filters' ) . '</label>' .
        '</div>' .

        '</div>'
        ;

        $html .= '<div class="awf-overlay-popup-row"><button type="button" title="' . esc_attr__( 'Save & Exit', 'annasta-filters' ) . '" class="button button-secondary awf-overlay-popup-save-btn" id="awf-save-custom-layout-btn">' . esc_html__( 'Save & Exit', 'annasta-filters' ) . '</button></div>';
      }

      $html .= '</div>';

      echo $html;
    }
    
    private function add_custom_layout_column() {
      $preset_id = intval( $_POST['awf_preset_id'] );
      $customizer_options = get_option( 'awf_customizer_options', array() );
      $settings = array( '', '', '' );

      if( ! empty( $customizer_options[ 'preset_' . $preset_id . '_custom_layout'] ) ) {
        $settings = explode( '|||', $customizer_options[ 'preset_' . $preset_id . '_custom_layout'] );
      }

      if( 3 !== count( $settings ) ) { $settings = array( '', '', '' ); }

      $columns = explode( '||', $settings[0] );
      $columns = array_values( array_filter( $columns ) );
      $columns[] = '25|%';
      $settings[0] = implode( '||', $columns );

      $customizer_options['preset_' . $preset_id . '_custom_layout'] = implode( '|||', $settings );

      update_option( 'awf_customizer_options', $customizer_options );

      $this->display_custom_layout_popup();
    }
    
    private function delete_custom_layout_column() {
      $preset_id = intval( $_POST['awf_preset_id'] );
      $delete_column = intval( $_POST['awf_delete_column'] ) - 1;
      $customizer_options = get_option( 'awf_customizer_options', array() );
      $settings = array();

      if( ! empty( $customizer_options[ 'preset_' . $preset_id . '_custom_layout'] ) ) {
        $settings = explode( '|||', $customizer_options[ 'preset_' . $preset_id . '_custom_layout'] );
      }

      if( 3 !== count( $settings ) ) { return; }

      $columns = explode( '||', $settings[0] );
      $columns = array_values( array_filter( $columns ) );

      if( isset( $columns[$delete_column] ) ) {
        unset( $columns[$delete_column] );
        $columns = array_values( $columns );
      }

      $settings[0] = implode( '||', $columns );

      $customizer_options['preset_' . $preset_id . '_custom_layout'] = implode( '|||', $settings );

      update_option( 'awf_customizer_options', $customizer_options );

      $this->display_custom_layout_popup();
    }
    
    private function update_custom_layout() {

      $preset_id = intval( $_POST['awf_preset_id'] );
      $customizer_options = get_option( 'awf_customizer_options', array() );
      $columns = array();
      $gap = array( '0', '%' );
      $wrap = '';

      $units = array( '%', 'px', 'em' );

      if( isset( $_POST['awf_custom_layout_columns'] ) ) {
        foreach( $_POST['awf_custom_layout_columns'] as $i => $column_settings ) {
          $column = array();
          $ci = intval( $i );

          if( empty( $column_settings['awf-custom-layout-autowidth-' . $ci] ) ) {
            $column[0] = abs( floatval( $column_settings['awf-custom-layout-width-' . $ci] ) );
          } else {
            $column[0] = 'auto';
          }
          if( in_array( $column_settings['awf-custom-layout-units-' . $ci], $units ) ) {
            $column[1] = $column_settings['awf-custom-layout-units-' . $ci];
          } else {
            $column[1] = '%';
          }

          $columns[] = implode( '|', $column );

          if( $ci > 25 ) { break; }
        }
      }

      if( isset( $_POST['awf_custom_layout_options'] ) ) {
        if( isset( $_POST['awf_custom_layout_options']['gap'] ) ) {
          $gap[0] = abs( floatval( $_POST['awf_custom_layout_options']['gap'] ) );
        }
        if( isset( $_POST['awf_custom_layout_options']['gap-units'] ) ) {
          if( in_array( $_POST['awf_custom_layout_options']['gap-units'], $units ) ) {
            $gap[1] = $_POST['awf_custom_layout_options']['gap-units'];
          } else {
            $gap[1] = '%';
          }
        }

        if( ! empty( $_POST['awf_custom_layout_options']['wrap'] ) ) {
          $wrap = 'yes';
        }
      }

      $columns = implode( '||', $columns );
      $gap = implode( '|', $gap );

      $customizer_options[ 'preset_' . $preset_id . '_custom_layout'] = implode( '|||', array( $columns, $gap, $wrap ) );

      update_option( 'awf_customizer_options', $customizer_options );
      $this->generate_styles_css();
    }
    
    public function add_range_slider_style_options( $options ) {
      $position = intval( array_search( 'minimalistic', array_keys( $options ) ) );
      $options = array_merge(
        array_slice( $options, 0, $position, true),
        array( 'rounded_3d' => __( 'Rounded 3D', 'annasta-filters' ), 'marker_3d' => __( 'Markers 3D', 'annasta-filters' ) ),
        array_slice( $options, $position, count( $options ) - $position, true)
      );

      $options['minimalistic_rounded'] = __( 'Minimalistic Rounded', 'annasta-filters' );
      $options['minimalistic_bars'] = __( 'Minimalistic Bars', 'annasta-filters' );

      return $options;
		}
    
    public function add_color_filter_style_options( $options ) {
      $options['round'] = __( 'Round', 'annasta-filters' );
      
      return $options;
    }
    
    public static function add_premium_customizer_sections( $sections ) {
      $position = intval( array_search( 'awf_filter_title_customizer', array_keys( $sections ) ) );
                
      if( ! empty( $position ) ) {
        $sections = array_merge(
          array_slice( $sections, 0, $position, true),
          array( 'awf_dropdown_customizer'  => __( 'Dropdowns', 'annasta-filters' ) ),
          array_slice( $sections, $position, count( $sections ) - 1, true)
        );
      }

      return $sections;
    }
    
    public static function get_image_filter_style_options() {
			return apply_filters( 'awf_image_filter_style_options', array(
        'square' => __( 'Square', 'annasta-filters' ),
        'round' => __( 'Round', 'annasta-filters' ),
        'plain' => __( 'Plain image', 'annasta-filters' ),
      ) );
		}

    protected function get_product_list_template_options() {

      $options = parent::get_product_list_template_options();

      $options['active_badges'] = __( 'Active filters badges', 'annasta-filters' );
      $options['reset_btn'] = __( 'Reset all filters button', 'annasta-filters' );

      return $options;
    }

    
    protected function get_product_list_template_option_hooks( $option ) {
      $hooks = parent::get_product_list_template_option_hooks( $option );

      if( 'active_badges' === $option ) {
        $hooks['js'] = __( 'Javascript', 'annasta-filters' );
      }

      return $hooks;
    }
    
    public function get_product_list_settings() {
			
      $settings = parent::get_product_list_settings();

      ksort( $settings );

			return $settings;
		}
    
    public function get_styles_settings() {

      $settings = parent::get_styles_settings();

      $settings[60] = array( 
        'id'       => 'awf_image_filter_style',
        'type'     => 'select',
        'options'  => A_W_F_premium_admin::get_image_filter_style_options(),
        'name'     => __( 'Image filter style', 'annasta-filters' ),
        'default'  => get_option( 'awf_image_filter_style', 'square' ),
      );

      ksort( $settings );

      return $settings;
    }
        
    public function get_templates_settings() {

      return array(
        array(
          'id' => 'awf_templates_titles_tab',
          'type' => 'title',
          'name' => __( 'Templates Manager', 'annasta-filters' ),
        ),
        
        array( 'type' => 'sectionend', 'id' => 'awf_templates_titles_tab' ),
      );
    }
    
    public function get_seo_settings() {

      return array(
        array(
          'id' => 'awf_seo_titles_tab',
          'type' => 'title',
          'name' => __( 'SEO Settings', 'annasta-filters' ),
        ),
        
        array( 
          'id'       => 'awf_page_title',
          'type'     => 'select',
          'options'  => array(
            'wc_default'    => __( 'Woocommerce default', 'annasta-filters' ),
            'awf_default'    => __( 'annasta Default page title', 'annasta-filters' ),
            'seo'        => __( 'Autogenerated list of annasta filters', 'annasta-filters' ),
          ),
          'name'     => __( 'Page title', 'annasta-filters' ),
          'default'  => get_option( 'awf_page_title', 'wc_default' ),
          'desc_tip' => __( 'Page (HTML document) title can be seen as the name of the page at the top of the browser window (or tab). It is also taken into account by the search engines indexing the pages of your shop.', 'annasta-filters' )
        ),
        
        array( 
          'id'       => 'awf_default_page_title',
          'type'     => 'text',
          'name'     => __( 'Default page title', 'annasta-filters' ),
          'default'  => get_option( 'awf_default_page_title', _x( 'Shop', 'annasta Default page title', 'annasta-filters' ) ),
          'desc_tip' => __( 'Choose the "annasta Default page title" in the "Page title" setting to display this string as a browser (HTML document) title for the pages of your shop. This title will also be used as page title whenever there are no active filters to display in the autogenerated filters lists.', 'annasta-filters' ),
        ),
        
        array( 
          'id'       => 'awf_shop_title',
          'type'     => 'select',
          'options'  => array(
            'wc_default'    => __( 'Woocommerce default', 'annasta-filters' ),
            'awf_default'    => __( 'annasta Default shop title', 'annasta-filters' ),
            'seo'        => __( 'Autogenerated list of annasta filters', 'annasta-filters' ),
          ),
          'name'     => __( 'Shop title', 'annasta-filters' ),
          'default'  => get_option( 'awf_shop_title', 'wc_default' ),
          'desc_tip' => __( 'The shop page heading above the products list can be left as is, changed with the help of the "annasta Default shop title" setting below, or get dynamically adjusted with each filters application, depending on the filters combination.', 'annasta-filters' )
        ),
        
        array( 
          'id'       => 'awf_default_shop_title',
          'type'     => 'text',
          'name'     => __( 'Default shop title', 'annasta-filters' ),
          'default'  => get_option( 'awf_default_shop_title', _x( 'Shop', 'annasta Default shop title', 'annasta-filters' ) ),
          'desc_tip' => __( 'Choose the "annasta Default shop title" in the "Shop title" setting to display this string as a heading in your shop pages. This title will also be used as shop title whenever there are no active filters to display in the autogenerated filters lists.', 'annasta-filters' ),
        ),
        
        array( 'type' => 'sectionend', 'id' => 'awf_seo_titles_tab' ),
        
        array(
          'id' => 'awf_seo_meta_description_tab',
          'type' => 'title',
          'name' => __( 'Meta Description', 'annasta-filters' ),
        ),

        array( 
          'id'       => 'awf_add_seo_meta_description', 
          'type'     => 'checkbox',
          'name'     => __( 'Add meta description', 'annasta-filters' ),
          'default'  => get_option( 'awf_add_seo_meta_description', 'no' ),
          'desc_tip' => __( 'Add the meta "description" tag to the pages of your shop.', 'annasta-filters' )
        ),
        
        array( 
          'id'       => 'awf_seo_meta_description',
          'type'     => 'textarea',
          'name'     => __( 'Meta description', 'annasta-filters' ),
          'default'  => stripcslashes( trim( get_option( 'awf_seo_meta_description', 'Browse our shop for {annasta_filters}!' ) ) ),
        ),
        
        array( 'type' => 'sectionend', 'id' => 'awf_seo_meta_description_tab' ),
        
        array(
          'id' => 'awf_seo_settings_tab',
          'type' => 'title',
          'name' => __( 'Filters List Generation', 'annasta-filters' ),
        ),
        
        array( 
          'id'       => 'awf_seo_filters_title_prefix',
          'type'     => 'text',
          'name'     => __( 'Filters prefix', 'annasta-filters' ),
          'default'  => get_option( 'awf_seo_filters_title_prefix', 'Shop for' ),
          'desc_tip' => __( 'String to add to title before the active filters list.', 'annasta-filters' ),
          'css'      => 'width: 200px;'
        ),
        
        array( 
          'id'       => 'awf_seo_filters_title_postfix',
          'type'     => 'text',
          'name'     => __( 'Filters postfix', 'annasta-filters' ),
          'default'  => get_option( 'awf_seo_filters_title_postfix', '' ),
          'desc_tip' => __( 'String to add after the active filters list.', 'annasta-filters' ),
          'css'      => 'width: 200px;'
        ),
        
        array( 
          'id'       => 'awf_seo_filters_separator',
          'type'     => 'text',
          'name'     => __( 'Filters separator', 'annasta-filters' ),
          'default'  => get_option( 'awf_seo_filters_separator', ' - ' ),
          'desc_tip' => __( 'Enter the string that you wish to be used between the different filter groups in the SEO adjusted page and shop title. An example of a title generated using the default value of " - " would be "Fruit, Berries - Red, Green, Purple - Small, Medium".', 'annasta-filters' ),
          'css'      => 'width: 100px;'
        ),
        
        array( 
          'id'       => 'awf_seo_filter_values_separator',
          'type'     => 'text',
          'name'     => __( 'Filter values separator', 'annasta-filters' ),
          'default'  => get_option( 'awf_seo_filter_values_separator', ', ' ),
          'desc_tip' => __( 'Choose a combination of characters to be used between the values of the same filter (for instance, multiple colors selected in a product colors filter). An example of a title created using the default value of ", " would be "Fruit, Berries - Red, Green, Purple - Small, Medium".', 'annasta-filters' ),
          'css'      => 'width: 100px;'
        ),
        
        array( 'type' => 'sectionend', 'id' => 'awf_seo_settings_tab' ),
      );
    }
    
    protected function update_seo_filter_settings( &$settings, $saved_settings, $filter ) {
      $id = 'awf_seo_filter_' . $filter . '_';
      
      $settings['enabled'] = isset( $_POST[$id . 'enabled_setting'] ) ? true : false;
      
      foreach( $settings as $setting => $value ) {
        if( in_array( $setting, array( 'enabled', 'position' ) ) ) { continue; }
        
        $name = $id . $setting . '_setting';
        
        if( 'labels' === $setting ) {
          foreach( $value as $label_name => $label ) {
            if( isset( $_POST[$name . '_' . $label_name] ) ) {
              $settings['labels'][$label_name] = sanitize_text_field( wp_unslash( $_POST[$name . '_' . $label_name] ) );
            }
          }
          
        } elseif( isset( $_POST[$name] ) ) {
          $settings[$setting] = sanitize_text_field( $this->convert_edge_spaces_to_nbsp( wp_unslash( $_POST[$name] ) ) );
        }

      }
    }
    
    protected function update_seo_filters_positions() {
      $positions = ( isset( $_POST['awf_seo_filters_positions'] ) && is_array( $_POST['awf_seo_filters_positions'] ) ) ?  array_map( 'sanitize_key', $_POST['awf_seo_filters_positions'] ) : array();
      
      $positions = array_flip( $positions );
      $positions = array_map( 'intval', $positions );
      
      $filters = get_option( 'awf_seo_filters_settings', array() );
      
      foreach( $filters as $filter => $settings ) {
        $id = 'awf_seo_filter_' . $filter . '_container';
        if( isset( $positions[$id] ) ) { $filters[$filter]['position'] = $positions[$id] + 1; }
      }

      uasort( $filters, function( $a, $b ) {
        return $a['position'] - $b['position'];
      } );
      
      update_option( 'awf_seo_filters_settings', $filters );
    }
    
    protected function display_premium_seo_settings() {
      
      $filters = $this->get_seo_filters_list();
      
      echo
      '<h3>' , esc_html__( 'Filters Settings', 'annasta-filters' ) , '</h3>',
      '<table class="widefat awf-seo-modules-list-table">',
      
      '<thead><tr>',
      '<th colspan="3"><span>' , esc_html__( 'Include filter', 'annasta-filters' ) , '</span><span class="woocommerce-help-tip" data-tip="' , esc_attr__( 'Check to use the filter for the title generation, uncheck to exclude. The meta description filters list is not affected by this option, it will contain all the active filters.', 'annasta-filters' ) , '"></span>' , '</th>',
      '<th><span>' , esc_html__( 'Prefix', 'annasta-filters' ) , '</span><span class="woocommerce-help-tip" data-tip="' , esc_attr__( 'String to display before filter values.', 'annasta-filters' ) , '"></span>' , '</th>',
      '<th><span>' , esc_html__( 'Filters labels', 'annasta-filters' ) , '</span><span class="woocommerce-help-tip" data-tip="' , esc_attr__( 'Adjust the filter terms\' labels as needed. If you wish to display a string in the title whenever the filter is not active, put it into the corresponding "Inactive" box.', 'annasta-filters' ) , '"></span>' , '</th>',
      '<th><span>' , esc_html__( 'Postfix', 'annasta-filters' ) , '</span><span class="woocommerce-help-tip" data-tip="' , esc_attr__( 'String to display after filter values.', 'annasta-filters' ) , '"></span>' , '</th>',
      '</tr></thead>',
      '<tbody>'
      ;
      
      $row_class = 'awf-odd';
      
      foreach( $filters as $filter => $settings ) {
        
        $id = 'awf_seo_filter_' . $filter . '_';
        $row_class = 'awf-odd' === $row_class ? 'awf-even' : 'awf-odd' ;
        
        if( 0 === strpos( $filter, 'taxonomy_' ) ) {
          $taxonomy_name = substr( $filter, strlen( 'taxonomy_' ) );
          $taxonomy = get_taxonomy( $taxonomy_name );
          if( $taxonomy ) { $settings['term_label'] = $taxonomy->label; }
          
        } elseif( 0 === strpos( $filter, 'meta_filter_' ) ) {
          $settings['term_label'] = sprintf( __( '%1$s Meta', 'annasta-filters' ), substr( $filter, strlen( 'meta_filter_' ) ) );
          
        } else {
          $settings['term_label'] = $this->get_filter_title( $filter );
        }
        
        if( ! empty( $settings['enabled'] ) ) { $enabled = ' checked="checked"'; } else { $enabled = ''; }
        
        echo
          '<tr class="' , $row_class , ' awf-seo-filter-settings-container" id="' , $id , 'container">',
          '<td class="sort-handle" title="' , esc_attr__( 'Move up or down to arrange the filters in order in which you want to see them in titles.', 'annasta-filters' ) ,
            '"></td>' , 
          '<td><input name="' , $id , 'enabled_setting" id="' , $id , 'enabled_setting" type="checkbox" class="" value="yes"' , $enabled , '></td>',
          '<td>' . $settings['term_label'] . '</td>',
          '<td><input name="' , $id , 'prefix_setting" id="' , $id , 'prefix_setting" type="text" value="' , esc_attr( $settings['prefix'] ) , '" placeholder="' , esc_attr__( 'prefix...', 'annasta-filters' ), '"></td>',
          '<td class="awf-seo-filter-terms-container">',
            '<table class="awf-seo-filter-terms-table">',
            '<tr>',
              '<td colspan="2"><input name="' , $id , 'empty_setting" id="' , $id , 'empty_setting" type="text" value="' , esc_attr( $settings['empty'] ) , '" class="awf-seo-filter-terms-inactive-setting" placeholder="' , esc_attr__( 'inactive filter string...', 'annasta-filters' ), '"></td>',
            '</tr>'
        ;
        
        if( 'stock' === $filter ) {
          $labels = $this->get_seo_filter_settings_defaults( 'stock' );
          foreach( $labels['labels'] as $slug => $label ) {
            echo
              '<tr>',
                '<td><label for="' , $id , 'labels_setting_' , $slug , '">' , esc_html( $label ) , '</label></td>',
                '<td><input name="' , $id , 'labels_setting_' , $slug , '" id="' , $id , 'labels_setting_' , $slug , '" type="text" value="' . esc_attr( $settings['labels'][$slug] ) , '"></td>',
              '</tr>'
            ;
          }
          
        } elseif( 'onsale' === $filter || 'featured' === $filter ) {
          $labels = $this->get_seo_filter_settings_defaults( $filter );
          echo
            '<tr>',
              '<td><label for="' , $id , 'labels_setting_yes">' , esc_html( $labels['labels']['yes'] ) , '</label></td>',
              '<td><input name="' , $id , 'labels_setting_yes" id="' , $id , 'labels_setting_yes" type="text" value="' , esc_attr( $settings['labels']['yes'] ) , '"></td>',
            '</tr>'
          ;
        }
        
        echo
          '</table></td>',
        
          '<td><input name="' , $id , 'postfix_setting" id="' , $id , 'postfix_setting" type="text" value="' , esc_attr( $settings['postfix'] ) , '" placeholder="' , esc_attr__( 'postfix...', 'annasta-filters' ), '"></td>',
          '</tr>'
        ;        
      }
      
      echo '</tbody></table>';
    }
    
    public function get_plugin_settings() {

      $settings = parent::get_plugin_settings();

      $filters_exceptions = A_W_F::$admin->get_all_filters();
      
      unset( $filters_exceptions['meta'] );
      
      foreach( $filters_exceptions as $exception => $label ) {
        if( strpos( $exception, 'taxonomy--' ) === 0 ) {
          unset( $filters_exceptions[$exception] );
          $exception = explode( 'taxonomy--', $exception );
          $exception = array_pop( $exception );
          if( $exception ) { $filters_exceptions[$exception] = $label; }
        }
      }

      $settings[70] = array( 
        'id'       => 'awf_hide_empty_filters', 
        'type'     => 'checkbox',
        'disabled' => true,
        'name'     => __( 'Hide empty filters', 'annasta-filters' ),
        'default'  => get_option( 'awf_hide_empty_filters', 'no' ),
        'desc_tip' => __( 'This option has been moved to the filter level, and is now available at the end of any taxonomy filter settings list, under the name of "Hide filter".', 'annasta-filters' )
      );

      $settings[160] = array(
        'id'       => 'awf_reset_all_exceptions', 
        'type'     => 'multiselect',
        'options'  => $filters_exceptions,
        'name'     => __( 'Reset button exceptions', 'annasta-filters' ),
        'default'  => get_option( 'awf_reset_all_exceptions', array( 'ppp', 'orderby' ) ),
        'desc_tip' => __( 'Select the filters and controls that you do not want to get effected by the "Clear all" button and reset filters.', 'annasta-filters' ),
        'class'      => 'chosen_select',
      );

      $settings[170] = array(
        'id'       => 'awf_badges_exceptions', 
        'type'     => 'multiselect',
        'options'  => $filters_exceptions,
        'name'     => __( 'Active badges exceptions', 'annasta-filters' ),
        'default'  => get_option( 'awf_badges_exceptions', array() ),
        'desc_tip' => __( 'No badges will be created for the filters and controls selected in this setting.', 'annasta-filters' ),
        'class'      => 'chosen_select',
      );

      $settings[180] = array(
        'id'       => 'awf_badge_reset_label', 
        'type'     => 'text',
        'name'     => __( 'Active badge reset tip', 'annasta-filters' ),
        'default'  => get_option( 'awf_badge_reset_label', __( '', 'annasta-filters' ) ),
        'desc_tip' => __( 'Customize hover tip for the active badge reset button. Leave blank if not needed.', 'annasta-filters' ),
      );

      $settings[250] = array( 'type' => 'title', 'id' => 'awf_advanced_plugin_settings_section' );
      $settings[260] = array( 'type' => 'awf_advanced_plugin_settings' );
      $settings[299] = array( 'type' => 'sectionend', 'id' => 'awf_advanced_plugin_settings_section' );

			ksort( $settings );
			
			return $settings;
    }
    
    public function display_advanced_plugin_settings() {
      
      $archive_identifier = get_option( 'awf_archive_identifier_pretty_name', 'archive-filter' );
      $pretty_names = $this->get_pretty_names_options();
      $taxonomies = $this->get_tax_var_names();
      $operators_select = array(
        'options' => array(
          'IN' => 'IN',
          'NOT IN' => 'NOT IN',
          'AND' => 'AND',
        )
      );

      echo
      '<table class="form-table awf-ts-h awf-ts-2 awf-ts-collapsed" data-ts="2">',
      '<tr>',
        '<td colspan="2" scope="row">', '<h3>', '<span>', esc_html__( 'Advanced Settings', 'annasta-filters' ), '</h3>', '</td>',
      '</tr>',
      '</table>'
      ;
      
      ?>
    <table class="form-table awf-advanced-plugin-settings-table awf-ts awf-ts-2 awf-ts-collapsed">

      <tr class="awf-archive-identifier-pretty-name-container">
        <th class="titledesc"><?php esc_html_e( 'Filterable archive identifier', 'annasta-filters' ); ?><span class="woocommerce-help-tip" data-tip="<?php esc_attr_e( 'This parameter appears on all the filtered archive pages.', 'annasta-filters' ); ?>"></span></th>
        <td><input type="text" name="awf_archive_identifier_pretty_name" value="<?php echo esc_attr( $archive_identifier ); ?>" style="width:300px;"><span class="awf-pretty-query-var-name-default" style="display: inline-block; padding: 0 8px; line-height: 33px;">archive-filter</span></td>
      </tr>

      <tr class="awf-pretty-query-var-names-container">
        <th class="titledesc"><?php esc_html_e( 'Query parameters\' labels', 'annasta-filters' ); ?><span class="woocommerce-help-tip" data-tip="<?php esc_attr_e( 'Define labels for the filters\' URL query parameters. Leave blank to use the default value.', 'annasta-filters' ); ?>"></span></th>
        <td>
          
          <div class="awf-comment">
            <p class="description">
              <?php wp_kses( _e( '<strong>WARNING: </strong>use this option with caution, making sure that the labels that you choose do not already exist as any of the Wordpress, Woocommerce, or any other plugin\'s native parameter names. Please use only the lowercase alphanumeric characters, dashes and underscores for the parameters\' labels.', 'annasta-filters' ), array( 'strong' => array() ) ); ?>
            </p>
          </div>
          
          <table>
            <?php foreach( $pretty_names as $var => $data ) : ?>
            <tr>
              <td><?php echo esc_html( $data['label'] ); ?></td>
              <td><input type="text" name="<?php echo esc_attr( $var ); ?>" value="<?php echo esc_attr( get_option( $var, '' ) ); ?>" class="awf-pretty-query-var-name"></td>
              <td class="awf-pretty-query-var-name-default"><?php echo $data['default']; ?></td>
            </tr>
            <?php endforeach; ?>
          </table>
        </td>
      </tr>

      <tr class="awf-tax-query-operators-container">
        <th class="titledesc"><?php esc_html_e( 'Query operators for product taxonomies', 'annasta-filters' ); ?><span class="woocommerce-help-tip" data-tip="<?php esc_attr_e( 'Change the default \'IN\' operator for taxonomy queries.', 'annasta-filters' ); ?>"></span></th>
        <td>
          <table>
            <?php foreach( $taxonomies as $tax => $label ) : ?>
            <tr>
              <td><?php echo esc_html( $label ); ?></td>
              <td>
                <?php
      $option = 'awf_' . $tax . '_query_operator';
      echo $this->build_select_html( array_merge( $operators_select, array( 'name' => $option, 'selected' => get_option( $option, 'in' ) ) ) );
                ?>
              </td>
            </tr>
            <?php endforeach; ?>
          </table>
        </td>
      </tr>
    </table>
      <?php
    }
    
    private function get_pretty_names_options() {
      $labels = array();
      $vars = get_option( 'awf_query_vars', array( 'tax' => array(), 'awf' => array(), 'range' => array(), 'meta' => array(), 'misc' => array() ) );
      
      $tax_labels = array();
      $taxonomies = get_object_taxonomies( 'product', 'objects' );
      foreach( $taxonomies as $t ) {
        if( in_array( $t->name, A_W_F::$excluded_taxonomies ) ) continue;

        $default = '';
        
        switch( $t->name ) {
          case 'product_cat':
            $default = 'product-categories';
            break;
          case 'product_tag':
            $default = 'product-tags';
            break;
          case 'pwb-brand':
            $default = 'filter-pwb-brand';
            break;
          default:
            $default = empty( $t->query_var ) ? $t->name . '-filter' : $t->query_var . '-filter';
            break;
        }
        
        $tax_labels[$t->name] = array( 'label' => $t->label,  'default' => $default );
      }
      
      $awf_labels = array(
        'search' => array( 'label' => __( 'Search', 'annasta-filters' ), 'default' => 's-filter' ),
        'ppp' => array( 'label' => __( 'Products per page', 'annasta-filters' ), 'default' => 'ppp' ),
        'stock' => array( 'label' => __( 'Availability', 'annasta-filters' ), 'default' => 'availability' ),
        'featured' => array( 'label' => __( 'Featured products', 'annasta-filters' ), 'default' => 'featured' ),
        'onsale' => array( 'label' => __( 'Sale products', 'annasta-filters' ), 'default' => 'onsale' ),
      );
      
      foreach( $vars as $type_name => $type_vars ) {
        if( 'range' === $type_name ) { continue; }
        
        foreach( $type_vars as $var => $var_name ) {
          switch( $type_name ) {
            case 'tax':
              if( isset( $tax_labels[$var] ) ) {
                $labels['awf_' . $var . '_pretty_name'] = $tax_labels[$var];
              }
              break;
            case 'awf':
              if( isset( $awf_labels[$var] ) ) {
                $labels['awf_' . $var . '_awf_module_pretty_name'] = $awf_labels[$var];
              }
              break;
            case 'meta':
              if( 'awf_date_filter_' === substr( $var, 0, 16) ) {
                $data = substr( $var, 16 );
                $p = strpos( $data, '_' ) + 1;

                $labels[$var . '_awf_meta_pretty_name'] = array( 'label' => sprintf( __( '%1$s Meta', 'annasta-filters' ), $var ), 'default' => substr( $data, $p ) . '-' . substr( $data, 0, 1 ) . '-date-filter' );

              } else {
                $labels['awf_' . $var . '_awf_meta_pretty_name'] = array( 'label' => sprintf( __( '%1$s Meta', 'annasta-filters' ), $var ), 'default' => $var . '-filter' );
              }
              break;
            default:
              break;
          }
        }
      }
      
      return $labels;
    }
    
    private function get_tax_var_names() {
      $tax_vars = array();
      
      $taxonomies = get_object_taxonomies( 'product', 'objects' );

      foreach( $taxonomies as $t ) {
        if( in_array( $t->name, A_W_F::$excluded_taxonomies ) ) continue;
        $tax_vars[$t->name] = $t->label;
      }
      
      return $tax_vars;
    }
    
    public function save_premium_settings() {
      
      global $current_section;
      
      if( 'plugin-settings' === $current_section ) {
        $pretty_names_options = $this->get_pretty_names_options();
        
        foreach( $pretty_names_options as $option => $data ) {
          $value = empty( $_POST[$option] ) ? '' : urldecode( sanitize_title( trim( $_POST[$option] ) ) );
          
          switch( $option ) {
            case 'awf_product_cat_pretty_name':
              if( empty( $value ) || 'product_cat' === $value ) { $value = 'product-categories'; }
              break;
            case 'awf_product_tag_pretty_name':
              if( empty( $value ) || 'product_tag' === $value ) { $value = 'product-tags'; }
              break;
            case 'awf_search_awf_module_pretty_name':
              if( 's' === $value ) { $value = ''; }
              break;
            default:
              break;
          }
          
          if( empty( $value ) ) {
            delete_option( $option );
          } else {
            update_option( $option, $value );
          }
        }

        if( isset( $_POST['awf_archive_identifier_pretty_name'] ) ) {
          $ai = sanitize_title( $_POST['awf_archive_identifier_pretty_name'] );
          if( empty( $ai ) ) {
            $ai = 'archive-filter';
          }

          update_option( 'awf_archive_identifier_pretty_name', $ai );
        }
        
        A_W_F::build_query_vars();
        
        $taxonomies = $this->get_tax_var_names();
        
        foreach( $taxonomies as $tax => $label ) {
          $option = 'awf_' . $tax . '_query_operator';
          
          if( empty( $_POST[$option] ) || ! in_array( $_POST[$option], array( 'IN', 'NOT IN', 'AND' ) )  ) { continue; }
          if( 'IN' === $_POST[$option] ) {
            delete_option( $option );
          } else {
            update_option( $option, $_POST[$option] );
          }
        }
        
        $this->clear_product_counts_cache();
      }
    }
    
    protected function update_premium_filter( $filter, $old_settings, &$response ) {

      if( isset( $_POST['awf_preset_' . $filter->preset_id . '_type'] ) && in_array( $_POST['awf_preset_' . $filter->preset_id . '_type'], array( 'sbs' ) ) ) {
        $filter->settings['reset_all'] = false;
      }

      if( isset( $filter->settings['height_limit'] ) ) {
        $filter->settings['style_options']['height_limit_style'] = sanitize_key( $this->get_sanitized_text_field_setting( $filter->prefix . 'height_limit_style', 'scrollbars' ) );
      }

      if( 'taxonomy' === $filter->module ) {
        $filter->settings['unhide_active_filter'] = $this->get_sanitized_checkbox_setting( $filter, 'unhide_active_filter' );
      }

      $filter->settings['is_collapsible'] = $filter->settings['is_dropdown'] = false;
      if( isset( $_POST[$filter->prefix . 'collapse_options'] ) && in_array( $_POST[$filter->prefix . 'collapse_options'], array( 'is_collapsible', 'is_dropdown' ) ) ) {
        $filter->settings[$_POST[$filter->prefix . 'collapse_options']] = true;
        $filter->settings['show_title'] = true;

        if( 'is_dropdown' === $_POST[$filter->prefix . 'collapse_options'] ) {

          $filter->settings['style_options']['disable_dropdown_autocollapse'] = $this->get_sanitized_checkbox_setting( $filter, 'disable_dropdown_autocollapse' );

          if( 'single' === $filter->settings['type'] ) {
            $filter->settings['active_dropdown_title'] = $this->get_sanitized_checkbox_setting( $filter, 'active_dropdown_title' );

          } else {
            $filter->settings['active_dropdown_title'] = false;
          }
        }
      }

      if( ! empty( $filter->settings['hierarchical_sbs'] ) && ! empty( $filter->settings['display_children'] ) ) {
        $filter->settings['display_children'] = false;
      }

      if( 'search' === $filter->module && ! empty( $filter->settings['autocomplete'] ) ) {
        $filter->settings['style_options']['autocomplete_view_all_label'] = $this->get_sanitized_text_field_setting( $filter->prefix . 'autocomplete_view_all_label' );
        $filter->settings['style_options']['autocomplete_height_limit'] = $this->get_sanitized_int_setting( $filter->prefix . 'autocomplete_height_limit' );
      }
      
      if( 'range' === $filter->settings['type'] && 'taxonomy_range' === $filter->settings['type_options']['range_type'] ) {
        $range_terms = $this->build_taxonomy_range_terms( $filter );
        
        $filter->settings['style_options']['slider_tooltips'] = 'none';
        $filter->settings['type_options']['range_values'] = 
        $filter->settings['type_options']['range_labels'] = array();
        
        $default_labels = wp_list_pluck( $range_terms['all_terms'], 'name', 'slug' );
        
        foreach( $range_terms['terms'] as $slug => $position ) {
          $id = $filter->prefix . 'range_term_' . $slug;
          if( isset( $_POST[$id] ) ) {
            $new_position = (int) $_POST[$id];
            
            $filter->settings['type_options']['range_values'][$slug] = $new_position;

            if( isset( $_POST[$id . '_label'] ) ) {
              $filter->settings['type_options']['range_labels'][$new_position] = $this->get_sanitized_text_field_setting( $id . '_label' );
            } else {
              $filter->settings['type_options']['range_labels'][$new_position] = $default_labels[$slug];
            }
          }
        }
        
        asort( $filter->settings['type_options']['range_values'] );
        
        ksort( $filter->settings['type_options']['range_labels'] );
        $filter->settings['type_options']['range_labels'] = array_combine( array_keys( $filter->settings['type_options']['range_values'] ), $filter->settings['type_options']['range_labels'] );
      }

      if( 'images' === $filter->settings['style'] ) {

        if( ! isset( $_POST[$filter->prefix . 'show_label'] ) ) { $filter->settings['style_options']['hide_label'] = true; }

        $one_image_on = $this->get_sanitized_checkbox_setting( $filter, 'one_image_on', false );

        if( ! empty( $one_image_on ) ) {
          $image_id = $image_id_url = '';

          if( ! empty( $_POST[$filter->prefix . 'one_image_id'] ) ) {
            $image_id = (int) $_POST[$filter->prefix . 'one_image_id'];
            $image_id_url = esc_url_raw( wp_get_attachment_url( $image_id ) );
          }

          if( isset( $_POST[$filter->prefix . 'one_image_url'] ) ) {
            $filter->settings['style_options']['one_image_url'] = esc_url_raw( trim( $_POST[$filter->prefix . 'one_image_url'] ) );
            if( $image_id_url !== $filter->settings['style_options']['one_image_url'] ) { $image_id = ''; }

          } else {
            $filter->settings['style_options']['one_image_url'] = $image_id_url;
          }

          $filter->settings['style_options']['one_image_id'] = $image_id;


        } else {
          $filter_terms = $filter->get_filter_terms( false );
          $filter->settings['style_options']['image_ids'] = $filter->settings['style_options']['image_urls'] = array();

          foreach( $filter_terms as $term ) {

            $image_id = $image_id_url = $url = '';

            if( isset( $_POST[$filter->prefix . 'image_id_' . $term->term_id] ) ) {
              $image_id = (int) $_POST[$filter->prefix . 'image_id_' . $term->term_id];
              $image_id_url = wp_get_attachment_url( $image_id );
            }

            if( isset( $_POST[$filter->prefix . 'image_url_' . $term->term_id] ) ) {
              $url = trim( $_POST[$filter->prefix . 'image_url_' . $term->term_id] );
              if( $image_id_url !== $url ) { $image_id = ''; }
            } else {
              $url = $image_id_url;
            }

            $filter->settings['style_options']['image_ids'][$term->term_id] = $image_id;
            $filter->settings['style_options']['image_urls'][$term->term_id] = esc_url_raw( $url );
          }
        }
        
      } elseif( 'custom-terms' === $filter->settings['style'] ) {
        $filter->settings['style_options']['term_icons'] = array();
        $filter->settings['style_options']['term_icons_hover'] = array();
        $filter->settings['style_options']['term_icons_solids'] = array();
        $filter->settings['style_options']['term_icons_hover_solids'] = array();
        $filter->settings['style_options']['term_labels'] = array();
        
        $terms = $filter->get_filter_terms();
        
        foreach( $terms as $i => $term ) {
          $filter->settings['style_options']['term_icons'][$term->term_id] = $this->get_sanitized_text_field_setting( $filter->prefix . 'term_' . $term->term_id . '_icon' );
          if( '' === $filter->settings['style_options']['term_icons'][$term->term_id] ) { $filter->settings['style_options']['term_icons'][$term->term_id] = ' '; }
          $filter->settings['style_options']['term_icons_solids'][$term->term_id] = $this->get_sanitized_checkbox_setting( $filter, 'term_' . $term->term_id . '_icon_solid' ) ? true : false;
          $filter->settings['style_options']['term_icons_hover'][$term->term_id] = $this->get_sanitized_text_field_setting( $filter->prefix . 'term_' . $term->term_id . '_icon_hover' );
          if( '' === $filter->settings['style_options']['term_icons_hover'][$term->term_id] ) { $filter->settings['style_options']['term_icons_hover'][$term->term_id] = ' '; }
          $filter->settings['style_options']['term_icons_hover_solids'][$term->term_id] = $this->get_sanitized_checkbox_setting( $filter, 'term_' . $term->term_id . '_icon_hover_solid' ) ? true : false;
          $filter->settings['style_options']['term_icons_active'][$term->term_id] = $this->get_sanitized_text_field_setting( $filter->prefix . 'term_' . $term->term_id . '_icon_active' );
          if( '' === $filter->settings['style_options']['term_icons_active'][$term->term_id] ) { $filter->settings['style_options']['term_icons_active'][$term->term_id] = ' '; }
          $filter->settings['style_options']['term_icons_active_solids'][$term->term_id] = $this->get_sanitized_checkbox_setting( $filter, 'term_' . $term->term_id . '_icon_active_solid' ) ? true : false;
          $filter->settings['style_options']['term_icons_active_hover'][$term->term_id] = $this->get_sanitized_text_field_setting( $filter->prefix . 'term_' . $term->term_id . '_icon_active_hover' );
          if( '' === $filter->settings['style_options']['term_icons_active_hover'][$term->term_id] ) { $filter->settings['style_options']['term_icons_active_hover'][$term->term_id] = ' '; }
          $filter->settings['style_options']['term_icons_active_hover_solids'][$term->term_id] = $this->get_sanitized_checkbox_setting( $filter, 'term_' . $term->term_id . '_icon_active_hover_solid' ) ? true : false;
          $filter->settings['style_options']['term_labels'][$term->term_id] = $this->get_sanitized_text_field_setting( $filter->prefix . 'term_' . $term->term_id . '_label' );
        }
        
      } elseif( 'range-stars' === $filter->settings['style'] ) {
        $terms = $filter->get_filter_terms();
        
        foreach( $terms as $i => $term ) {
          $filter->settings['style_options']['term_labels'][$term->term_id] = $this->get_sanitized_text_field_setting( $filter->prefix . 'term_' . $term->term_id . '_label' );
        }

      } elseif( 'range-slider' === $filter->settings['style'] ) {
        $filter->settings['style_options']['hide_slider'] = $this->get_sanitized_checkbox_setting( $filter, 'hide_slider' );
        $filter->settings['style_options']['hide_slider_labels'] = $this->get_sanitized_checkbox_setting( $filter, 'hide_slider_labels' );
        $filter->settings['style_options']['hide_slider_poles'] = $this->get_sanitized_checkbox_setting( $filter, 'hide_slider_poles' );
        
      }
    }
    
    public function build_hide_empty_select( $filter ) {
      $options['name'] = $options['id'] = $filter->prefix . 'hide_empty';
      $options['class'] = 'awf-hide-empty-select';
      $options['selected'] = $filter->settings['hide_empty'];

      $options['options']['none'] = __( 'Regular display', 'annasta-filters' );
      $options['options']['hidden'] = __( 'Hidden', 'annasta-filters' );
      $options['options']['css'] = __( 'Lower opacity', 'annasta-filters' );
      $options['options']['disabled'] = __( 'Disabled', 'annasta-filters' );

      return $this->build_select_html( $options );
    }

    public function build_images_options_html( $filter ) {

      $one_image_id = $one_image_url = '';
      $one_image_preview = 'data:image/gif;base64,R0lGODlhAQABAIAAAAAAAP///yH5BAEAAAAALAAAAAABAAEAAAIBRAA7';
      $one_image_checked = $terms_hidden = '';
      $one_image_media_handler = ' style="display: none;"';

      if( ! empty( $filter->settings['style_options']['one_image_url'] ) ) {
        $one_image_url = $one_image_preview = esc_url( $filter->settings['style_options']['one_image_url'] );
        $one_image_id = $filter->settings['style_options']['one_image_id'];
        $one_image_checked = ' checked="checked"';
        $terms_hidden = ' style="display: none;"';
        $one_image_media_handler = '';
      }

      $html = '<ul>';
      $html .= '<li>';
      $html .= '<input id="' . $filter->prefix . 'show_label" type="checkbox" name="' . $filter->prefix . 'show_label" value="yes"';
      if( ! isset( $filter->settings['style_options']['hide_label'] ) ) { $html .= ' checked="checked"'; }
      $html .= '>';
      $html .= '<label for="' . $filter->prefix . 'show_label">' . esc_html__( 'Display label', 'annasta-filters' ) . '</label>';
      $html .= '</li>';

      $html .= '<li>';
      $html .= '<input id="' . $filter->prefix . 'one_image_on" type="checkbox" name="' . $filter->prefix . 'one_image_on" value="yes" class="awf-one-image-on"' . $one_image_checked . '>';
      $html .= '<label for="' . $filter->prefix . 'one_image_on">' . esc_html__( 'One image for all terms', 'annasta-filters' ) . '</label>';
      $html .= '</li>';

      $html .= '<li class="awf-one-image-container"' . $one_image_media_handler . '>';
      $html .= '<input type="hidden" name="' . $filter->prefix . 'one_image_id" value="' . esc_attr( $one_image_id ) . '" class="awf-image-id"/>';
      $html .= '<img src="' . $one_image_preview . '" class="awf-image-preview"/>';

      $html .= '<div class="awf-image-url-wrapper"><div class="awf-image-url-container">';
      $html .= '<label for="' . $filter->prefix . 'one_image_url" class="awf-image-url-label">' . esc_html__( 'Image link', 'annasta-filters' ) . '</label>';
      $html .= '<input type="text" name="' . $filter->prefix . 'one_image_url" value="' . esc_url( $one_image_url ) . '" class="awf-one-image-url">';
      $html .= '</div></div>';
      $html .= '<button type="button" class="button button-secondary awf-icon awf-media-handler-btn awf-one-image-handler" title="' . esc_attr__( 'Select image from media library', 'annasta-filters' ) . '"></button>';
      $html .= '</li>';
      $html .= '</ul>';

      $terms_by_parent = $filter->build_terms_by_parent( $filter->get_filter_terms() );

      $html .= '<table class="awf-terms-images-container awf-filter-options-secondary-table"' . $terms_hidden . '><tbody>';
      $html .= $this->build_terms_image_list( $filter, $terms_by_parent );
      $html .= '</tbody></table>';

      return $html;
    }
    
    protected function build_terms_image_list( $filter, $terms_by_parent, $parent_id = 0 ) {
      $terms_html = '';

      foreach ( $terms_by_parent[$parent_id] as $term ) {

        $url = $preview_url = $image_id = '';
        $sanitized_term_id = sanitize_html_class( $term->term_id );

        if( isset( $filter->settings['style_options']['image_urls'] ) && ! empty( $filter->settings['style_options']['image_urls'][$term->term_id] ) ) {
          $url = $preview_url = esc_url( $filter->settings['style_options']['image_urls'][$term->term_id] );
          $image_id = $filter->settings['style_options']['image_ids'][$term->term_id];
        } else {
          $preview_url = 'data:image/gif;base64,R0lGODlhAQABAIAAAAAAAP///yH5BAEAAAAALAAAAAABAAEAAAIBRAA7';
        }

        $terms_html .= '<tr class="awf-image-term-container">';
        $terms_html .= '<td>' . esc_html( $term->name ) . '</td>';
        $terms_html .= '<td>';
        $terms_html .= '<input type="hidden" name="' . $filter->prefix . 'image_id_' . $sanitized_term_id . '" value="' . esc_attr( $image_id ) . '"';
        $terms_html .= ' class="awf-image-id"/>';
        $terms_html .= '<img src="' . $preview_url . '" class="awf-image-preview"/>';
        $terms_html .= '</td>';
        $terms_html .= '<td class="awf-image-url-wrapper"><div class="awf-image-url-container">';
        $terms_html .= '<label for="' . $filter->prefix . 'image_url_' . $sanitized_term_id . '" class="awf-image-url-label">' . esc_html__( 'Image link', 'annasta-filters' ) . '</label>';
        $terms_html .= '<input type="text" name="' . $filter->prefix . 'image_url_' . $sanitized_term_id . '" id="' . $filter->prefix . 'image_url_' . $sanitized_term_id . '" value="' . esc_url( $url ) . '" class="awf-image-url">';
        $terms_html .= '</div></td>';
        $terms_html .= '<td><button type="button" class="button button-secondary awf-icon awf-media-handler-btn" title="' . esc_attr__( 'Select image from media library', 'annasta-filters' ) . '"></button></td>';
        $terms_html .= '</tr>';

        if( isset( $terms_by_parent[$term->term_id] ) ) {
          $terms_html .= $this->build_terms_image_list( $filter, $terms_by_parent, $term->term_id );
        }
      }

      return $terms_html;
    }

    public function build_custom_terms_options_html( $filter ) {
      $terms = $filter->get_filter_terms();

      $html = '<table class="awf-custom-terms-container">';
      
      $html .= '<thead>';
      $html .= '<tr>';
      $html .= '<th>' . esc_html__( 'Filter Terms', 'annasta-filters' ) . '</th>';
      $html .= '<th>' . esc_html__( 'Icon', 'annasta-filters' ) . '</th>';
      $html .= '<th>' . esc_html__( 'Solid', 'annasta-filters' ) . '</th>';
      $html .= '<th>' . esc_html__( 'Hover Icon', 'annasta-filters' ) . '</th>';
      $html .= '<th>' . esc_html__( 'Solid', 'annasta-filters' ) . '</th>';
      $html .= '<th>' . esc_html__( 'Active Icon', 'annasta-filters' ) . '</th>';
      $html .= '<th>' . esc_html__( 'Solid', 'annasta-filters' ) . '</th>';
      $html .= '<th>' . esc_html__( 'Active Hover', 'annasta-filters' ) . '</th>';
      $html .= '<th>' . esc_html__( 'Solid', 'annasta-filters' ) . '</th>';
      $html .= '<th>' . esc_html__( 'Label (optional)', 'annasta-filters' ) . '</th>';
      $html .= '</tr>';
      $html .= '</thead>';
      
      $html .= '<tbody>';
      
      foreach( $terms as $i => $term ) {
        $sanitized_term_id = sanitize_html_class( $term->term_id );
        
        if( isset( $filter->settings['style_options']['term_icons'] ) ) {
          $v = empty( $filter->settings['style_options']['term_icons'][$term->term_id] ) ? '' : $filter->settings['style_options']['term_icons'][$term->term_id];
          $vh = empty( $filter->settings['style_options']['term_icons_hover'][$term->term_id] ) ? '' : $filter->settings['style_options']['term_icons_hover'][$term->term_id];
          $va = empty( $filter->settings['style_options']['term_icons_active'][$term->term_id] ) ? '' : $filter->settings['style_options']['term_icons_active'][$term->term_id];
          $vah = empty( $filter->settings['style_options']['term_icons_active_hover'][$term->term_id] ) ? '' : $filter->settings['style_options']['term_icons_active_hover'][$term->term_id];
          $l = empty( $filter->settings['style_options']['term_labels'][$term->term_id] ) ? '' : $filter->settings['style_options']['term_labels'][$term->term_id];
          $classes = empty( $filter->settings['style_options']['term_icons_solids'][$term->term_id] ) ? '' : ' awf-solid';
          $hover_classes = empty( $filter->settings['style_options']['term_icons_hover_solids'][$term->term_id] ) ? '' : ' awf-solid';
          $active_classes = empty( $filter->settings['style_options']['term_icons_active_solids'][$term->term_id] ) ? '' : ' awf-solid';
          $active_hover_classes = empty( $filter->settings['style_options']['term_icons_active_hover_solids'][$term->term_id] ) ? '' : ' awf-solid';
          
        } else {
          $v = $vh = $va = $vah = '';
          $l = $term->name;
          $classes = $hover_classes = $active_classes = $active_hover_classes = '';
        }
        
        $html .= '<tr>';
        $html .= '<td>' . esc_html( $term->name ) . '</td>';
        $html .= '<td class="awf-filter-icons-column-dark-l"><input type="text" name="' . $filter->prefix . 'term_' . $sanitized_term_id . '_icon" value="' . esc_attr( $v ) . '" class="awf-filter-icon awf-term-filter-icon' . $classes . '"></td>';
        $html .= '<td class="awf-filter-icons-column-dark-r"><input type="checkbox" name="' . $filter->prefix . 'term_' . $sanitized_term_id . '_icon_solid" value="yes" class="awf-solid-icon"' . ( empty( $classes ) ? '' : ' checked="checked"' ) . '></td>';
        $html .= '<td><input type="text" name="' . $filter->prefix . 'term_' . $sanitized_term_id . '_icon_hover" value="' . esc_attr( $vh ) . '" class="awf-filter-icon-hover' . $hover_classes . '"></td>';
        $html .= '<td><input type="checkbox" name="' . $filter->prefix . 'term_' . $sanitized_term_id . '_icon_hover_solid" value="yes" class="awf-solid-icon-hover"' . ( empty( $hover_classes ) ? '' : ' checked="checked"' ) . '></td>';
        $html .= '<td class="awf-filter-icons-column-dark-l"><input type="text" name="' . $filter->prefix . 'term_' . $sanitized_term_id . '_icon_active" value="' . esc_attr( $va ) . '" class="awf-filter-icon-active' . $active_classes . '"></td>';
        $html .= '<td class="awf-filter-icons-column-dark-r"><input type="checkbox" name="' . $filter->prefix . 'term_' . $sanitized_term_id . '_icon_active_solid" value="yes" class="awf-solid-icon-active"' . ( empty( $active_classes ) ? '' : ' checked="checked"' ) . '></td>';
        $html .= '<td><input type="text" name="' . $filter->prefix . 'term_' . $sanitized_term_id . '_icon_active_hover" value="' . esc_attr( $vah ) . '" class="awf-filter-icon-active-hover' . $active_hover_classes . '"></td>';
        $html .= '<td><input type="checkbox" name="' . $filter->prefix . 'term_' . $sanitized_term_id . '_icon_active_hover_solid" value="yes" class="awf-solid-icon-active-hover"' . ( empty( $active_hover_classes ) ? '' : ' checked="checked"' ) . '></td>';
        $html .= '<td class="awf-filter-icons-column-dark-l"><input type="text" name="' . $filter->prefix . 'term_' . $sanitized_term_id . '_label" value="' . esc_attr( $l ) . '"></td>';
        $html .= '</tr>';
      }
      
      $html .= '</tbody></table>';
      
      $html .= '<div class="awf-icons-examples-container" data-tip="' . esc_attr__( 'Copied to clipboard', 'annasta-filters' ) . '">';
      $html .= '<h4>' . esc_html__( 'Click an icon to copy to clipboard, then paste to the chosen box', 'annasta-filters' );
      $html .= '<span class="woocommerce-help-tip" data-tip="' . esc_attr__( 'Some icons are available only in the solid version, so make sure to toggle the \'Solid style\' checkbox if the icon doesn\'t display properly. Go to Fontawesome Icons Gallery for more amazing icons for your shop!', 'annasta-filters' ) . '"></span>' . '</h4>';
      
      $html .= '<span class="awf-icon-example"></span>';
      $html .= '<span class="awf-icon-example"></span>';
      $html .= '<span class="awf-icon-example"></span>';
      $html .= '<span class="awf-icon-example"></span>';
      $html .= '<span class="awf-icon-example"></span>';
      $html .= '<span class="awf-icon-example"></span>';
      $html .= '<span class="awf-icon-example"></span>';
      $html .= '<span class="awf-icon-example awf-solid" title="' . esc_attr__( 'Solid style only', 'annasta-filters' ) . '"></span>';
      $html .= '<span class="awf-icon-example awf-solid" title="' . esc_attr__( 'Solid style only', 'annasta-filters' ) . '"></span>';
      $html .= '<span class="awf-icon-example"></span>';
      $html .= '<span class="awf-icon-example"></span>';
      $html .= '<span class="awf-icon-example"></span>';
      $html .= '<span class="awf-icon-example"></span>';
      $html .= '<span class="awf-icon-example"></span>';
      $html .= '<span class="awf-icon-example"></span>';
      $html .= '<span class="awf-icon-example awf-solid" title="' . esc_attr__( 'Solid style only', 'annasta-filters' ) . '"></span>';
      $html .= '<span class="awf-icon-example awf-solid" title="' . esc_attr__( 'Solid style only', 'annasta-filters' ) . '"></span>';
      $html .= '<span class="awf-icon-example awf-solid" title="' . esc_attr__( 'Solid style only', 'annasta-filters' ) . '"></span>';
      $html .= '<span class="awf-icon-example awf-solid" title="' . esc_attr__( 'Solid style only', 'annasta-filters' ) . '"></span>';
      $html .= '<span class="awf-icon-example awf-solid" title="' . esc_attr__( 'Solid style only', 'annasta-filters' ) . '"></span>';
      $html .= '<span class="awf-icon-example"></span>';
      $html .= '<span class="awf-icon-example awf-solid" title="' . esc_attr__( 'Solid style only', 'annasta-filters' ) . '"></span>';
      $html .= '<span class="awf-icon-example awf-solid" title="' . esc_attr__( 'Solid style only', 'annasta-filters' ) . '"></span>';
      $html .= '<span class="awf-icon-example awf-solid" title="' . esc_attr__( 'Solid style only', 'annasta-filters' ) . '"></span>';
      $html .= '<span class="awf-icon-example awf-solid" title="' . esc_attr__( 'Solid style only', 'annasta-filters' ) . '"></span>';
      $html .= '<span class="awf-icon-example awf-solid" title="' . esc_attr__( 'Solid style only', 'annasta-filters' ) . '"></span>';
      $html .= '<span class="awf-icon-example awf-solid" title="' . esc_attr__( 'Solid style only', 'annasta-filters' ) . '"></span>';
      
      $html .= '</div>';

      return $html;
    }

    public function build_taxonomy_range( $filter ) {
      
      $range_terms = $this->build_taxonomy_range_terms( $filter );

      if( count( $range_terms['terms'] ) < 2 ) { return '<div style="text-align:center;"><p>' . esc_html__( 'At least 2 filter items are needed for the range control creation. Please create more terms for this taxonomy (or adjust the excluded filter items) and try again.', 'annasta-filters' ) . '</p></div>'; }

      $html = '<div><p class="awf-info-notice">' . esc_html__( 'Sort the range items from the smallest (on top) to the largest (in the bottom), and assign labels for the range filters.', 'annasta-filters' ) . '</p></div>';

      $html .= '<table class="awf-filter-options-secondary-table awf-taxonomy-range-container" style="margin:2em 0 0;">';  
      $html .= '<tbody>';
      $html .= '<tr>';
          $html .= '<td rowspan="' . ( count( $range_terms['terms'] ) + 2 ) . '" style="height: 80px;min-height: 80px;padding: 0 20px 0 10px;">';
          $html .= '<div style="display:flex;flex-direction: column;height: 100%;">';
          $html .= '<div style="padding: 10px 0;">min</div>';
          $html .= '<div style="position:relative;margin:0;height: 100%;min-height: 100px;width: 49%;border-right:1px solid #cccccc;">';
          $html .= '<div style="position:absolute;right:-6px;bottom:-8px;font-family:\'AWF FA\',\'Font Awesome 5 Free\';font-size:16px;font-weight:900;"></div></div>';
          $html .= '<div style="padding: 10px 0;">max</div>';
          $html .= '</div>';
          $html .= '</td><td colspan="3" style="height: 0;min-height:0;"></td>';
      $html .= '</tr>';
      
      $old_labels = isset( $filter->settings['type_options']['range_labels'] ) ? $filter->settings['type_options']['range_labels'] : array();
      $filter->settings['type_options']['range_values'] = $filter->settings['type_options']['range_labels'] = array();
      $filter->settings['style_options']['slider_tooltips'] = 'none';
      
      foreach( $range_terms['terms'] as $slug => $position ) {

        $term = $range_terms['all_terms'][$range_terms['terms_slugs'][$slug]];
        
        $filter->settings['type_options']['range_values'][$slug] = $position;
        $filter->settings['type_options']['range_labels'][$slug] = isset( $old_labels[$slug] ) ? $old_labels[$slug] : $term->name;
        
        $id = $filter->prefix . 'range_term_' . esc_attr( $slug );

        $html .= '<tr id="' . $id . '_container" class="awf-range-term-container">';
        $html .= '<td class="sort-handle" title="' . esc_attr__( 'Move up or down to arrange the terms in the ascending order.', 'annasta-filters' ) . '"><span>' . esc_html( $position ) . '</span>';
        $html .= '<input type="hidden" name="' . $id . '" value="' . esc_attr( $position ) . '" >';
        $html .= '</td>';
        $html .= '<td style="text-align:right;">';
        $html .= sprintf( esc_html__( '%1$s label:', 'annasta-filters' ), esc_html( $term->name ) ) . ' ' . '</td>';
        $html .= '<td><input type="text" name="' . $id . '_label" value="' . esc_attr( $filter->settings['type_options']['range_labels'][$slug] ) . '"></td>';
        $html .= '</tr>';

      }

      $html .= '</tbody></table>';

      return $html;
    }

    protected function build_taxonomy_range_terms( $filter ) {
      $range_terms = array();

      $range_terms['all_terms'] = $filter->get_limited_terms();

      $range_terms['all_terms'] = array_filter( $range_terms['all_terms'], function( $term ) {
        if ( 0 === $term->parent ) {
          return true;
        }
        return false;
      });
      
      $range_terms['terms_slugs'] = wp_list_pluck( $range_terms['all_terms'], 'slug' );
      $range_terms['terms_slugs'] = array_flip( $range_terms['terms_slugs'] );

      if( ! empty( $filter->settings['type_options']['range_values'] ) ) {
        $terms_to_remove = array_diff_key( $filter->settings['type_options']['range_values'], $range_terms['terms_slugs'] );
        $terms_to_add = array_diff_key( $range_terms['terms_slugs'], $filter->settings['type_options']['range_values'] );
        
        $range_terms['terms'] = array_diff_key( $filter->settings['type_options']['range_values'], $terms_to_remove );
        $range_terms['terms'] = $range_terms['terms'] + $terms_to_add; /* ! array_merge doesn't work for numeric slugs ! */

      } else {
        $range_terms['terms'] = $range_terms['terms_slugs'];
      }
      
      $position = 1;
      
      foreach( $range_terms['terms'] as $slug => $value ) {
        $range_terms['terms'][$slug] = $position++;
      }
      
      return $range_terms;
    }

    public function build_range_stars_options_html( $filter ) {
      $terms = $filter->get_filter_terms();
      
      $html = '<table class="awf-range-stars-container">';
      
      $html .= '<thead>';
      $html .= '<tr>';
      $html .= '<th>' . esc_html__( 'Range', 'annasta-filters' ) . '</th>';
      $html .= '<th>' . esc_html__( 'Stars Preview', 'annasta-filters' ) . '</th>';
      $html .= '<th>' . esc_html__( 'Label', 'annasta-filters' ) . '</th>';
      $html .= '</tr>';
      $html .= '</thead>';
      
      $html .= '<tbody>';
      
      foreach( $terms as $i => $term ) {
        if( isset( $filter->settings['style_options']['term_labels'] ) ) {
          $label = empty( $filter->settings['style_options']['term_labels'][$term->term_id] ) ? '' : $filter->settings['style_options']['term_labels'][$term->term_id];
          
        } else {
          $label = $term->name;
        }
        
        $html .= '<tr>';
        $html .= '<td>' . esc_html( $term->name ) . '</td>';
        $html .= '<td><span class="awf-stars-preview';
        
        $html .= ' awf-' . floor( $term->next_value ) . '-stars"></span></td>';
        $html .= '<td><input type="text" name="' . $filter->prefix . 'term_' . sanitize_html_class( $term->term_id ) . '_label" value="' . esc_attr( $label ) . '"></td>';
        $html .= '</tr>';
      }
      
      $html .= '</tbody></table>';
      
      return $html;
    }
    
    protected function add_premium_tooltips_options( $filter, &$tooltips_options ) {
      if( 'taxonomy_range' !== $filter->settings['type_options']['range_type'] ) {
        $tooltips_options['interactive_above'] = __( 'Interactive tooltips above range slider', 'annasta-filters' );
      }

      $tooltips_options['below'] = __( 'Display "min - max" values below slider', 'annasta-filters' );
    }
    
    protected function get_interactive_tooltips_html() {
      $html = '<div class="awf-interactive-slider-tooltips-container" >';
      $html .= '<div class="awf-interactive-tooltip-container"><input class="awf-interactive-tooltip-0" type="text" data-handle="0" /></div>';
      $html .= '<div class="awf-interactive-tooltip-container"><input class="awf-interactive-tooltip-1" type="text" data-handle="1" /></div>';
      $html .= '</div>';
      
      return $html;
    }
  
    protected function update_range_positions( $filter ) {
      $positions = array_map( 'sanitize_title', $_POST['awf_range_positions'] );
      $positions = array_map( 'urldecode', $positions );
      $values = array_flip( $positions );
      $labels = array_map( 'sanitize_text_field', wp_unslash( $_POST['awf_range_labels'] ) );

      $range_terms = $this->build_taxonomy_range_terms( $filter );

      if( empty( array_diff_key( $values, $range_terms['terms'] ) ) && ( count( $values ) === count( $labels ) ) ) {
        $filter->settings['type_options']['range_values'] = array();
        $filter->settings['type_options']['range_labels'] = array();
        $position = 1;
        
        foreach( $values as $slug => $i ) {

          $filter->settings['type_options']['range_values'][$slug] = $position++;
          $filter->settings['type_options']['range_labels'][$slug] = $labels[$i];
        }
        
        update_option( $filter->prefix. 'settings', $filter->settings );
        
      } else {
        wp_send_json_error( array( 'awf_error_message' => __( 'An error has occured when sorting the range positions. Please reload this page and try again.', 'annasta-filters' ) ), 400 );
      }
    }

    public function display_premium_autocomplete_options( $filter ) {
?>
                    <div class="awf-ac-options-row">
                      <label for="<?php echo $filter->prefix; ?>autocomplete_view_all_label"><?php esc_html_e( 'View all results link text', 'annasta-filters' ); ?></label>
                      <input type="text" name="<?php echo $filter->prefix; ?>autocomplete_view_all_label" id="<?php echo $filter->prefix; ?>autocomplete_view_all_label" value="<?php if( ! empty( $filter->settings['style_options']['autocomplete_view_all_label'] ) ) { echo esc_attr( $filter->settings['style_options']['autocomplete_view_all_label'] ); } else { esc_attr_e( 'View all results', 'annasta-filters' ); } ?>">
                    </div>

                    <div class="awf-ac-options-row">
                      <label for="<?php echo $filter->prefix; ?>autocomplete_height_limit">
                        <?php esc_html_e( 'Autocomplete height limit (pixels)', 'annasta-filters' ); ?>
                        <span class="woocommerce-help-tip" data-tip="<?php esc_attr_e( 'Scrollbars will be added to the autocomplete box whenever the height of the list of suggested products exceeds this setting in pixels. Set to zero to disable.', 'annasta-filters' ); ?>"></span>
                      </label>
                      <input type="text" name="<?php echo $filter->prefix; ?>autocomplete_height_limit" id="<?php echo $filter->prefix; ?>autocomplete_height_limit" value="<?php echo esc_attr( empty( $filter->settings['style_options']['autocomplete_height_limit'] ) ? '0' : $filter->settings['style_options']['autocomplete_height_limit'] ); ?>" style="width:80px;min-width:80px;">
                    </div>
<?php
    }

    protected function get_taxonomy_defaults( $filter_data ) {
      $settings = array(
        'taxonomy'                => $filter_data['taxonomy']->name,
        'title'                   => $filter_data['taxonomy']->label,
        'show_title'              => true,
        'button_submission'       => false,
        'active_prefix'           => '',
        'show_active'             => false,
        'reset_active'            => false,
        'reset_active_label'      => _x( 'Clear filters', 'Label for single filter reset button', 'annasta-filters' ),
        'block_deselection'       => false,
        'reset_all'               => false,
        'force_reload'            => false,
        'redirect_to_archive'     => false,
        'is_collapsible'          => false,
        'collapsed_on'            => false,
        'is_dropdown'             => false,
        'active_dropdown_title'   => false,
        'type'                    => 'multi',
        'type_options'            => array(),
        'style'                   => 'icons',
        'style_options'           => array(),
        'show_in_row'             => false,
        'show_search'             => false,
        'show_search_placeholder' => '',
        'terms_limitation_mode'   => 'exclude',
        'excluded_items'          => array(),
        'included_items'          => array(),
        'sort_by'                 => 'admin',
        'sort_order'              => 'none',
        'height_limit'            => '0',
        'shrink_height_limit'     => true,
        'show_count'              => false,
        'hide_empty'              => 'none',
        'hide_filter'             => '',
      );

      if( $filter_data['taxonomy']->hierarchical ) {
        $position = intval( array_search( 'show_search', array_keys( $settings ) ) );
        
        $settings = array_merge(
          array_slice( $settings, 0, $position, true),
          array(
            'hierarchical_level' => 1,
            'display_children' => true,
            'children_collapsible' => true,
            'children_collapsible_on' => true,
            'hierarchical_sbs' => false,
          ),
          array_slice( $settings, $position, count( $settings ) - 1, true)
        );
      }

      return $settings;
    }

    protected function get_search_defaults( $filter_data ) {
      return( array(
        'title'                   => $filter_data['title'],
        'show_title'              => true,
        'button_submission'       => false,
        'submit_on_change'        => false,
        'active_prefix'           => '',
        'show_active'             => false,
        'reset_all'               => false,
        'is_collapsible'          => false,
        'collapsed_on'            => false,
        'type'                    => null,
        'type_options'            => array(),
        'style'                   => null,
        'style_options'           => array(),
        'placeholder'             => __( 'Search products...', 'annasta-filters' ),
        'autocomplete'            => false,
        'height_limit'            => '0',
        'shrink_height_limit'     => true,
      ) );
    }

    protected function get_price_defaults( $filter_data ) {
      return( array(
        'title'                   => $filter_data['title'],
        'show_title'              => true,
        'button_submission'       => false,
        'active_prefix'           => '',
        'show_active'             => false,
        'reset_active'            => false,
        'reset_active_label'      => _x( 'Clear filters', 'Label for single filter reset button', 'annasta-filters' ),
        'reset_all'               => false,
        'is_collapsible'          => false,
        'collapsed_on'            => false,
        'is_dropdown'             => false,
        'active_dropdown_title'   => false,
        'type'                    => 'range',
        'type_options'            => array(
          'range_type'    => 'auto_range',
          'range_values'  => array( floatval( 0 ), floatval( $this->get_products_max_price() ) ),
          'decimals'      => intval( 0 )
        ),
        'style'                   => 'range-slider',
        'style_options'           => array( 'step' => floatval( 1 ), 'value_prefix' => '', 'value_postfix' => '' ),
        'show_in_row'             => false,
        'height_limit'            => '0',
        'shrink_height_limit'     => true,
      ) );
    }

    protected function get_stock_defaults( $filter_data ) {
      return( array(
        'title'                   => $filter_data['title'],
        'show_title'              => true,
        'button_submission'       => false,
        'active_prefix'           => '',
        'show_active'             => false,
        'reset_active'            => false,
        'reset_active_label'      => _x( 'Clear filters', 'Label for single filter reset button', 'annasta-filters' ),
        'reset_all'               => false,
        'is_collapsible'          => false,
        'collapsed_on'            => false,
        'is_dropdown'             => false,
        'active_dropdown_title'   => false,
        'type'                    => 'single',
        'style'                   => 'radios',
        'style_options'           => array(),
        'show_in_row'             => false,
        'show_search'             => false,
        'show_search_placeholder' => '',
        'terms_limitation_mode'   => 'exclude',
        'excluded_items'          => array(),
        'included_items'          => array(),
        'height_limit'            => '0',
        'shrink_height_limit'     => true,
      ) );
    }

    protected function get_featured_defaults( $filter_data ) {
      return( array(
        'title'                 => $filter_data['title'],
        'show_title'            => true,
        'button_submission'     => false,
        'reset_all'             => false,
        'is_collapsible'        => false,
        'collapsed_on'          => false,
        'is_dropdown'           => false,
        'type'                  => 'multi',
        'style'                 => 'checkboxes',
        'style_options'         => array(),
        'height_limit'          => '0',
        'shrink_height_limit'   => true,
      ) );
    }

    protected function get_onsale_defaults( $filter_data ) {
      return( array(
        'title'                 => $filter_data['title'],
        'show_title'            => true,
        'button_submission'     => false,
        'reset_all'             => false,
        'is_collapsible'        => false,
        'collapsed_on'          => false,
        'is_dropdown'           => false,
        'type'                  => 'multi',
        'style'                 => 'checkboxes',
        'style_options'         => array(),
        'height_limit'          => '0',
        'shrink_height_limit'   => true,
      ) );
    }

    protected function get_rating_defaults( $filter_data ) {
      return( array(
        'title'                   => $filter_data['title'],
        'show_title'              => true,
        'button_submission'       => false,
        'active_prefix'           => '',
        'show_active'             => false,
        'reset_active'            => false,
        'reset_active_label'      => _x( 'Clear filters', 'Label for single filter reset button', 'annasta-filters' ),
        'reset_all'               => false,
        'is_collapsible'          => false,
        'collapsed_on'            => false,
        'is_dropdown'             => false,
        'type'                    => 'range',
        'type_options'            => array( 
          'range_type' => 'custom_range',
          'range_values' => array( round( floatval( 1 ), 2, PHP_ROUND_HALF_UP ), round( floatval( 2 ), 2, PHP_ROUND_HALF_UP ), round( floatval( 3 ), 2, PHP_ROUND_HALF_UP ), round( floatval( 4 ), 2, PHP_ROUND_HALF_UP ), round( floatval( 5 ), 2, PHP_ROUND_HALF_UP ), round( floatval( 5.01 ), 2, PHP_ROUND_HALF_UP ) ),
          'precision' => round( floatval( 0.01 ), 2, PHP_ROUND_HALF_UP ),
          'decimals' => intval( 0 ),
        ),
        'style'                   => 'radios',
        'style_options'           => array( 'step' => floatval( 1 ), 'value_prefix' => '', 'value_postfix' => '' ),
        'show_in_row'             => false,
        'show_search'             => false,
        'show_search_placeholder' => '',
        'height_limit'            => '0',
        'shrink_height_limit'     => true,
      ) );
    }

    protected function get_ppp_defaults( $filter_data ) {
      return( array(
        'title'                   => $filter_data['title'],
        'show_title'              => true,
        'button_submission'       => false,
        'show_active'             => false,
        'reset_active'            => false,
        'reset_active_label'      => _x( 'Clear filters', 'Label for single filter reset button', 'annasta-filters' ),
        'reset_all'               => false,
        'is_collapsible'          => false,
        'collapsed_on'            => false,
        'is_dropdown'             => false,
        'active_dropdown_title'   => false,
        'type'                    => 'single',
        'style'                   => 'radios',
        'style_options'           => array(),
        'ppp_values'              => array( 12 => __( 'products per page', 'annasta-filters' ) ),
        'show_in_row'             => false,
        'show_search'             => false,
        'show_search_placeholder' => '',
        'height_limit'            => '0',
        'shrink_height_limit'     => true,
      ) );
    }
    
    protected function get_orderby_defaults( $filter_data ) {
      return( array(
        'title'                   => $filter_data['title'],
        'show_title'              => true,
        'button_submission'       => false,
        'active_prefix'           => '',
        'show_active'             => false,
        'reset_active'            => false,
        'reset_active_label'      => _x( 'Clear filters', 'Label for single filter reset button', 'annasta-filters' ),
        'reset_all'               => false,
        'is_collapsible'          => false,
        'collapsed_on'            => false,
        'is_dropdown'             => false,
        'active_dropdown_title'   => false,
        'type'                    => 'single',
        'style'                   => 'radios',
        'style_options'           => array(),
        'show_in_row'             => false,
        'show_search'             => false,
        'show_search_placeholder' => '',
        'terms_limitation_mode'   => 'exclude',
        'excluded_items'          => array(),
        'included_items'          => array(),
        'height_limit'            => '0',
        'shrink_height_limit'     => true,
      ) );
    }
    
    protected function get_meta_defaults( $filter_data ) {
      return( array(
        'meta_name'               => '',
        'title'                   => $filter_data['title'],
        'show_title'              => true,
        'button_submission'       => false,
        'active_prefix'           => '',
        'show_active'             => false,
        'reset_active'            => false,
        'reset_active_label'      => _x( 'Clear filters', 'Label for single filter reset button', 'annasta-filters' ),
        'reset_all'               => false,
        'is_collapsible'          => false,
        'collapsed_on'            => false,
        'is_dropdown'             => false,
        'active_dropdown_title'   => false,
        'type'                    => 'single',
        'type_options'            => array(),
        'style'                   => 'radios',
        'style_options'           => array(),
        'show_in_row'             => false,
        'show_search'             => false,
        'show_search_placeholder' => '',
        'terms_limitation_mode'   => 'exclude',
        'excluded_items'          => array(),
        'included_items'          => array(),
        'height_limit'            => '0',
        'shrink_height_limit'     => true,
      ) );
    }

    protected function generate_premium_customizer_css( $options ) {
      $css = '';
			
			$value = isset( $options['awf_color_filter_style'] ) ? $options['awf_color_filter_style'] : get_option( 'awf_color_filter_style' );
			
      if( 'round' === $value ) {
        $css .= '.awf-style-colours label::before,.awf-style-colours label::after,.awf-style-colours .awf-filter-container .awf-count-wo-label .awf-filter-count{border-radius: 50%;}';
      } else {
				if( isset( $options['awf_color_filter_style'] ) ) {
          /* Reset to square for the Customizer preview */
					$css .= '.awf-style-colours label::before,.awf-style-colours label::after,.awf-style-colours .awf-filter-container .awf-count-wo-label .awf-filter-count{border-radius: 0;}';
				}
			}
			
			$value = isset( $options['awf_image_filter_style'] ) ? $options['awf_image_filter_style'] : get_option( 'awf_image_filter_style' );
			
			$reset_images_css = <<<'CSS_CODE'
.awf-style-images .awf-filter-container label::before {
  left: 0;
  top: 0;
  width: 40px;
  height: 40px;
  border: 1px solid #cccccc;
	border-radius: 0;
  box-shadow: inset 0px 0px 0px 1px rgba(0,0,0,0.02);
}

.awf-style-images .awf-filter-container.awf-active label::before,
.awf-style-images .awf-filter-container.awf-active:not(.awf-hover-off) label:hover::before {
  border: 1px solid #999999;
  box-shadow: inset 0px 0px 0px 1px rgba(0,0,0,0.5);
}

.awf-style-images a:focus label::before,
.awf-style-images .awf-filter-container.awf-active a:focus label::before {
  border: 1px solid #999999;
  box-shadow: none;
}

.awf-style-images .awf-filter-container label::after {
  left: 4px;
  top: 4px;
  width: 34px;
  height: 34px;
  border: 1px solid #f1f1f1;
  box-sizing: border-box;
	opacity: 0.9;
  background-position: left top;
  background-size: cover;
	border-radius: 0;
}

.awf-style-images .awf-filter-container .awf-count-wo-label .awf-filter-count{ border-radius: 0; }

CSS_CODE;
			
      if( 'round' === $value ) {
        $css .= $reset_images_css . '.awf-style-images .awf-filter-container label::before,.awf-style-images .awf-filter-container.awf-active label::before,.awf-style-images .awf-filter-container.awf-active:not(.awf-hover-off) label:hover::before,.awf-style-images .awf-filter-container label::after,.awf-style-images .awf-filter-container .awf-count-wo-label .awf-filter-count{border-radius: 50%;}';
        
      } elseif( 'plain' === $value ) {
        $css .= <<<'CSS_CODE'
.awf-style-images .awf-filter-container label::before,
.awf-style-images .awf-filter-container.awf-active label::before,
.awf-style-images .awf-filter-container.awf-active:not(.awf-hover-off) label:hover::before{background:none;border:none;border-radius:0;box-shadow:none;}
.awf-style-images .awf-filter-container label::after{background-position:center;background-size:contain;border:none;border-radius:0;box-shadow:none;}
CSS_CODE;
				
      } else {
				if( isset( $options['awf_image_filter_style'] ) ) {
           /* Reset to square for the Customizer preview */
					$css .= $reset_images_css;
				}
			}

      $value = isset( $options['awf_range_slider_style'] ) ? $options['awf_range_slider_style'] : get_option( 'awf_range_slider_style', 'minimalistic' );

      switch( $value ) {
        
        case 'rounded_3d':
          $css .= '.awf-filters-container .awf-filter-container.awf-range-slider-container,.awf-filters-container .awf-taxonomy-range-slider-container{font-size:12px;}.noUi-horizontal{height:10px;}.noUi-connect{background:#eb2222;}.noUi-horizontal .noUi-handle{top:-12px;right:-14px;height:28px;width:29px;border-radius:50%;}.noUi-horizontal .noUi-handle::before,.noUi-horizontal .noUi-handle::after{display:none;}.noUi-pips-horizontal{margin-top:5px;}.noUi-marker-horizontal.noUi-marker-large{height:10px;}.noUi-horizontal .noUi-tooltip{margin-bottom:5px;line-height:12px;background:#fbfbfb;}.noUi-horizontal .noUi-tooltip::before{content:"\f0d7";position:absolute;display:block;top:auto;bottom:0;left:50%;margin-bottom:-7.5px;transform:translate(-50%, 0);line-height:12px;color:#ccc;font-family: "AWF FA","Font Awesome 5 Free";font-size:12px;font-weight:900;}';
          $css .= '.awf-slider-tooltips-below{margin-top:40px;}';

          break;

        case( 'minimalistic_rounded' ):
          $css .= <<<'CSS_CODE'
.noUi-horizontal { height: 5px; }

.noUi-target {
  background: #ccc;
  border: none;
  border-radius: 0;
  box-shadow: none;
}

.noUi-connect { background: #666; }

.noUi-horizontal .noUi-handle {
  top: -8px;
  right: -10px;
  height: 20px;
  width: 20px;
  background: #666;
  box-shadow: none;
  border: none;
  border-radius: 50%;
}

.noUi-horizontal .noUi-handle::before,
.noUi-horizontal .noUi-handle::after {
  content: none;
  display: none;
}

.noUi-pips-horizontal {
  margin-top: -16px;
}

.noUi-marker-horizontal.noUi-marker-large {
  height: 7px;
  background: #999;
  border-radius: 0;
}

.noUi-marker-horizontal.noUi-marker-normal{display:none;}

CSS_CODE;

          break;
          
        case( 'minimalistic_bars' ):
          $css .= <<<'CSS_CODE'
.noUi-horizontal {
  height: 5px;
}

.noUi-target {
  background: #ccc;
  border: none;
  border-radius: 0;
  box-shadow: none;
}

.noUi-connect {
  background: #666;
}

.noUi-horizontal .noUi-handle {
  top: -6px;
  right: -17px;
  height: 28px;
  width: 34px;
  border: none;
  background: transparent;
  box-shadow: none;
}

.noUi-horizontal .noUi-handle::before {
  content: "";
  display: block;
  position: absolute;
  top: 1px;
  left: 16px;
  width: 3px;
  height: 15px;
  background: #666;
  z-index: 1;
}

.noUi-pips-horizontal {
  margin-top: -16px;
}

.noUi-marker-horizontal.noUi-marker-large {
  height: 7px;
  background: #999;
}

.noUi-marker-horizontal.noUi-marker-normal{display:none;}

CSS_CODE;

          break;

        case 'marker_3d':
          $css .= '.awf-filters-container .awf-filter-container.awf-range-slider-container,.awf-filters-container .awf-taxonomy-range-slider-container{font-size:12px;}.noUi-horizontal{height:10px;}.noUi-connect{background:#eb2222;}
          .noUi-horizontal .noUi-handle{top:-5px;right:-8px;height:18px;width:15px;background:linear-gradient(0deg, #fff 0%, #fff 10%, rgba(231, 231, 231, 1) 100%);box-shadow: inset 0 0 3px #fff, inset 0 -3px 3px #fff, inset 0 0px 10px #eee, inset 0 0px 5px #ccc, 0 0px 2px 0px #ddd, 0px 0px 1px 0px #737373;
          }
          .noUi-horizontal .noUi-handle::before{top:6px;left:4px;height:6px;width:4px;background:transparent;border-left:1px solid #929292;border-right: 1px solid #a6a6a6;box-shadow: 1px 0px 0px #fff, inset 1px 0 0 #fff;}
          .noUi-horizontal .noUi-handle::after{content: "\f0d7";position:absolute;display:block;top:11.5px;left:-0.5px;width:0px;transform:translate(-50%, 0);line-height:12px;color: #fff;font-family: "AWF FA", "Font Awesome 5 Free";font-size:23px;font-weight:900;text-shadow: 0 1px 1px #ccc, 0 1px 5px #efefef, 0px -2px 3px #fff, 0px -2px 3px #fff;border:none;}
          .noUi-pips-horizontal{margin-top:5px;}.noUi-marker-horizontal.noUi-marker-large{height:10px;}.noUi-horizontal .noUi-tooltip{margin-bottom:5px;line-height:12px;background:#fbfbfb;}.noUi-horizontal .noUi-tooltip::before{content:"\f0d7";position:absolute;display:block;top:auto;bottom:0;left:50%;margin-bottom:-7.5px;transform:translate(-50%, 0);line-height:12px;color:#ccc;font-family: "AWF FA","Font Awesome 5 Free";font-size:12px;font-weight:900;}';
          $css .= '.awf-slider-tooltips-below{margin-top:40px;}';

          break;

        default: break;
      }

      foreach( A_W_F::$presets as $preset_id => $preset ) {
        if( 'custom' === get_option( 'awf_preset_' . $preset_id . '_layout', '1-column' ) && ! empty( $options['preset_' . $preset_id . '_custom_layout'] ) ) {
          $settings = explode( '|||', $options[ 'preset_' . $preset_id . '_custom_layout'] );
          if( 3 !== count( $settings ) ) { continue; }
  
          $columns = explode( '||', $settings[0] );
          $columns = array_values( array_filter( $columns ) );
          $n = count( $columns );
  
          foreach( $columns as $i => $c ) {
            $co = explode( '|', $c );
            if( 'auto' === $co[0] ) {
              $w = 'auto';
            } else {
              $w = sanitize_text_field( implode( '', $co ) );
            }
            $css .= '.awf-preset-wrapper.awf-preset-' . $preset_id . '-wrapper > form.awf-filters-form > div:nth-child(' . $n . 'n + ' . ($i + 1 ) . ') {flex-basis:' . $w . ';}';
          }

          $gap = sanitize_text_field( implode( '', explode( '|', $settings[1] ) ) );
          $css .= '.awf-preset-wrapper.awf-preset-' . $preset_id . '-wrapper > form.awf-filters-form{column-gap:' . $gap . ';}';

          if( ! empty( $settings[2] ) ) {
            $css .= '.awf-preset-wrapper.awf-preset-' . $preset_id . '-wrapper > form.awf-filters-form{flex-wrap:nowrap;}';
          }
        }
      }
      
      return $css;
		}

    protected function generate_premium_css() { return ''; }

    protected function generate_premium_filter_css( $filter, $languages = array() ) {
      $css = '';

      if( 'search' === $filter->module && ! empty( $filter->settings['style_options']['autocomplete_height_limit'] ) ) {
        $css .= '.awf-filters-' . $filter->preset_id . '-' . $filter->id . '-container .awf-product-search-autocomplete-container {position:relative;';
        $css .= 'max-height:' . $filter->settings['style_options']['autocomplete_height_limit'] . 'px;overflow:hidden;';
        if ( 'yes' !== get_option( 'awf_pretty_scrollbars' ) ) { $css .= 'overflow-y:auto;'; }
        $css .= '}';
      }
            
      if( 'custom-terms' === $filter->settings['style'] && isset( $filter->settings['style_options']['term_icons'] ) ) {
        $filter_terms = $filter->get_filter_terms();

        foreach( $filter_terms as $i => $term ) {
          $term_css = '';

          $term_css .= '.awf-filters-' . $filter->preset_id . '-' . $filter->id . '-container .awf-term-' .  $term->term_id . '::before{content:"';
          if( ! empty( $filter->settings['style_options']['term_icons'][$term->term_id] ) ) {
            $term_css .= $filter->settings['style_options']['term_icons'][$term->term_id];
          }
          $term_css .= '";';

          if( ! empty( $filter->settings['style_options']['term_icons_solids'][$term->term_id] ) ) {
            $term_css .= 'font-weight:900;';
          } else { $term_css .= 'font-weight:inherit;'; }
          $term_css .= '}';

          if( ! empty( $filter->settings['style_options']['term_icons_hover'][$term->term_id] ) ) {
            $term_css .= '.awf-filters-' . $filter->preset_id . '-' . $filter->id . '-container .awf-term-' .  $term->term_id . ':hover::before{content:"' . $filter->settings['style_options']['term_icons_hover'][$term->term_id] . '";';
            if( ! empty( $filter->settings['style_options']['term_icons_hover_solids'][$term->term_id] ) ) {
              $term_css .= 'font-weight:900;';
            } else { $term_css .= 'font-weight:inherit;'; }
            $term_css .= '}';
          }
          if( ! empty( $filter->settings['style_options']['term_icons_active'][$term->term_id] ) ) {
            $term_css .= '.awf-filters-' . $filter->preset_id . '-' . $filter->id . '-container .awf-active .awf-term-' .  $term->term_id . '::before{content:"' . $filter->settings['style_options']['term_icons_active'][$term->term_id] . '";';
            if( ! empty( $filter->settings['style_options']['term_icons_active_solids'][$term->term_id] ) ) {
              $term_css .= 'font-weight:900;';
            } else { $term_css .= 'font-weight:inherit;'; }
            $term_css .= '}';
          }
          if( ! empty( $filter->settings['style_options']['term_icons_active_hover'][$term->term_id] ) ) {
            $term_css .= '.awf-filters-' . $filter->preset_id . '-' . $filter->id . '-container .awf-active .awf-term-' .  $term->term_id . ':hover::before{content:"' . $filter->settings['style_options']['term_icons_active_hover'][$term->term_id] . '";';
            if( ! empty( $filter->settings['style_options']['term_icons_active_hover_solids'][$term->term_id] ) ) {
              $term_css .= 'font-weight:900;';
            } else { $term_css .= 'font-weight:inherit;'; }
            $term_css .= '}';
          }

          if( ! empty( $languages ) && isset( $filter->settings['taxonomy'] ) ) {
            $languages_css = '';
            $term_class = '.awf-term-' .  $term->term_id;

            foreach( $languages as $language ) {
              $language_term_id = apply_filters( 'wpml_object_id', $term->term_id, $filter->settings['taxonomy'], TRUE, $language );

              if( $language_term_id !== $term->term_id ) {
                $languages_css .= str_replace( $term_class, '.awf-term-' . $language_term_id, $term_css );
              }
            }
            $css .= $languages_css;
          }

          $css .= $term_css;
        }
        
      } else if( 'images' === $filter->settings['style'] ) {
        if( ! empty( $filter->settings['style_options']['one_image_url'] ) ) {
          $css .= '.awf-filters-' . $filter->preset_id . '-' . $filter->id . '-container.awf-style-images label::after {';
          $css .= 'background-image:url(' . $filter->settings['style_options']['one_image_url'] . ')}';

        } else {
          if( isset( $filter->settings['style_options']['image_urls'] ) ) {
            foreach( $filter->settings['style_options']['image_urls'] as $term_id => $image_url ) {
              $css .= '.awf-filters-' . $filter->preset_id . '-' . $filter->id . '-container.awf-style-images label.awf-term-' . $term_id . '::after {';
              $css .= 'background-image:url(' . $image_url . ');}';

              if( ! empty( $languages ) && isset( $filter->settings['taxonomy'] ) ) {
                foreach( $languages as $language ) {
                  $language_term_id = apply_filters( 'wpml_object_id', $term_id, $filter->settings['taxonomy'], TRUE, $language );
                  if( $language_term_id !== $term_id ) {
                    $css .= '.awf-filters-' . $filter->preset_id . '-' . $filter->id . '-container.awf-style-images label.awf-term-' . $language_term_id . '::after {';
                    $css .= 'background-image:url(' . $image_url . ');}';
                  }
                }
              }
            }
          }
        }
      } else if( 'range-slider' === $filter->settings['style'] ) {

        $stbmt = 0;

        if( ! empty( $filter->settings['style_options']['hide_slider'] ) ) {
          $css .= '.awf-filters-' . $filter->preset_id . '-' . $filter->id . '-container .awf-filter-container.awf-range-slider-container{display:none;}';
          $css .= '.awf-filters-' . $filter->preset_id . '-' . $filter->id . '-container .awf-interactive-slider-tooltips-container{margin-bottom:25px;}';
        }

        if( ! empty( $filter->settings['style_options']['hide_slider_labels'] ) ) {
          $stbmt += 10;
          $css .= '.awf-filters-' . $filter->preset_id . '-' . $filter->id . '-container .awf-filter-container.awf-range-slider-container .noUi-value.noUi-value-large{display:none;}';
        }

        if( ! empty( $filter->settings['style_options']['hide_slider_poles'] ) ) {
          $stbmt += 10;
          $css .= '.awf-filters-' . $filter->preset_id . '-' . $filter->id . '-container .awf-filter-container.awf-range-slider-container .noUi-pips .noUi-marker {display:none;}';
          if( ! in_array( get_option( 'awf_range_slider_style', 'minimalistic' ), array( 'minimalistic', 'minimalistic_rounded', 'minimalistic_bars') )  )
          $css .= '.awf-filters-' . $filter->preset_id . '-' . $filter->id . '-container .awf-filter-container.awf-range-slider-container .noUi-value{margin-top:-15px;}';
        }

        if( ! empty( $stbmt ) ) {
          $css .= '.awf-filters-' . $filter->preset_id . '-' . $filter->id . '-container .awf-filter-container.awf-range-slider-container .awf-slider-tooltips-below{margin-top:' . ( 40 - $stbmt ) . 'px;}';
        }
      }

      if( ! empty( $filter->settings['height_limit'] ) ) {
        if( ! empty( $filter->settings['is_dropdown'] ) && ! empty( $filter->settings['button_submission'] ) && 'range-slider' !== $filter->settings['style'] ) {
          $css .= '.awf-dropdown.awf-button-filter .awf-' . $filter->preset_id . '-' . $filter->id . '-submit-btn-container {top:' . ( intval( $filter->settings['height_limit'] ) - 13 ) . 'px;}';
        }

        if( ! empty( $filter->settings['style_options']['height_limit_style'] ) && 'toggle' === $filter->settings['style_options']['height_limit_style'] ) {
          $css .= '.awf-thl.awf-thl-expanded:not(.awf-collapsed) .awf-filters-' . $filter->preset_id . '-' . $filter->id . '-container {max-height:inherit;height:inherit;}';
        }
      }

      return $css;
    }

    public function display_custom_templates_settings() {
      $filters = A_W_F::$admin->get_all_filters();
      $templates = $this->get_templates();

      $presets_list = '';
      foreach( $templates['presets'] as $template_id => $template ) {
        $presets_list .= '<div class="awf-settings-tb-row awf-template-row" data-preset-template-id="' . $template_id . '">';
        
        $presets_list .= '<div class="awf-template-name">' . esc_html( $template['template_name'] ) . '</div>';

        $presets_list .= '<div class="awf-template-version">';
        if( $template['awf_version'] !== A_W_F_VERSION ) {
          $presets_list .= '<span class="woocommerce-help-tip" data-tip="' . esc_attr__( 'This template has been created in a different version of annasta Filters, and may be incompatible with some current plugin options.', 'annasta-filters' ) . '"></span>';
          $presets_list .= '<span title="">' . esc_html( $template['awf_version'] ) . '<span>';
        }
        $presets_list .= '</div>';

        $presets_list .= '<div class="awf-settings-tb-row-buttons-container">';
        $presets_list .= '<button type="button" title="' . esc_attr__( 'Rename template', 'annasta-filters' ) . '" class="button button-secondary awf-fa-icon awf-fas-icon awf-popup-rename-preset-template-btn"></button>';
        $presets_list .= '<button type="button" title="' . esc_attr__( 'Export template to JSON file', 'annasta-filters' ) . '" class="button button-secondary awf-fa-icon awf-fas-icon awf-export-preset-template-btn"></button>';
        $presets_list .= '<button type="button" title="' . esc_attr__( 'Delete template', 'annasta-filters' ) . '" class="button button-secondary awf-fa-icon awf-fas-icon awf-fa-delete-btn awf-delete-preset-template-btn" data-preset-template-id="' . $template_id . '"></button>';
        $presets_list .= '</div>';

        $presets_list .= '</div>';
      }

      $filter_name = false;
      $filters_list = '';
      if( isset( $_GET['awf-filter'] ) && isset( $filters[$_GET['awf-filter']] ) ) {
        $filter_name = $_GET['awf-filter'];
      }

      $templates_by_filter = array();
      foreach( $templates['filters'] as $template_id => $template ) {
        $templates_by_filter[$template['name']][] = $template_id;

        if( $template['name'] !== $filter_name ) { continue; }

        if( 'meta' === $template['module'] ) {
          if( empty( $template['settings']['meta_name'] ) ) {
            continue;
          }
        }

        $filters_list .= '<div class="awf-settings-tb-row awf-template-row" data-filter-template-id="' . $template_id . '">';
        
        $filters_list .= '<div class="awf-template-name">' . esc_html( $template['template_name'] ) . '</div>';

        $filters_list .= '<div class="awf-template-version">';
        if( $template['awf_version'] !== A_W_F_VERSION ) {
          $filters_list .= '<span class="woocommerce-help-tip" data-tip="' . esc_attr__( 'This template has been created in the older version of annasta Filters, and may not support all current options.', 'annasta-filters' ) . '"></span>';
          $filters_list .= '<span title="">' . esc_html( $template['awf_version'] ) . '<span>';
        }
        $filters_list .= '</div>';

        $filters_list .= '<div class="awf-settings-tb-row-buttons-container">';
        $filters_list .= '<button type="button" title="' . esc_attr__( 'Rename template', 'annasta-filters' ) . '" class="button button-secondary awf-fa-icon awf-fas-icon awf-popup-rename-filter-template-btn"></button>';
        $filters_list .= '<button type="button" title="' . esc_attr__( 'Export template to JSON file', 'annasta-filters' ) . '" class="button button-secondary awf-fa-icon awf-fas-icon awf-export-filter-template-btn"></button>';
        $filters_list .= '<button type="button" title="' . esc_attr__( 'Delete template', 'annasta-filters' ) . '" class="button button-secondary awf-fa-icon awf-fas-icon awf-fa-delete-btn awf-delete-filter-template-btn" data-filter-template-id="' . $template_id . '"></button>';
        $filters_list .= '</div>';

        $filters_list .= '</div>';
    }

      $filters_select = array(
        'id' => 'awf-templates-filters-select',
        'selected' => $filter_name ? $filter_name : 'none',
        'options' => array(
          'none' => __( 'Select filter', 'annasta-filters' )
        )
      );

      foreach( $filters as $name => $label ) {
        if( isset( $templates_by_filter[$name] ) ) {
          $filters_select['options'][$name] = $label . ' (' . count( $templates_by_filter[$name] ) . ')';
        }
      }

      $filters_select_html = $this->build_select_html( $filters_select );

      echo
      '<div class="awf-info-notice-container"><span class="awf-info-notice">',
      sprintf( wp_kses( __( 'Please consult the <a href="%1$s" target="_blank">plugin documentation</a> to learn about <strong>templates limitations</strong>, especially when applying templates imported from different configurations!', 'annasta-filters' ), array(  'a' => array( 'href' => array(),'target' => array() ), 'strong' => array() ) ), esc_url( 'https://annasta.net/plugins/annasta-woocommerce-product-filters/documentation/templates-settings/templates-manager/' ) ),
      '</div>',

      '<div class="awf-templates-subheader">',
      '<h3>', esc_html__( 'Preset templates', 'annasta-filters' ), '</h3>',
      '<div class="awf-settings-tb-row-buttons-container">',
      '<button type="button" title="', esc_attr__( 'Create preset template', 'annasta-filters' ), '" class="button button-secondary awf-fa-icon-text-btn awf-popup-create-preset-template-btn">' . esc_html__( 'Create', 'annasta-filters' ) . '</button>',
      '<button type="button" title="' . esc_attr__( 'Import template from JSON file', 'annasta-filters' ) . '" class="button button-secondary awf-fa-icon-text-btn awf-import-preset-template-btn">' . esc_html__( 'Import', 'annasta-filters' ) . '</button>',
      '<input type="file" id="awf-import-json-preset-template" accept=".json" style="display:none;" />',
      '</div>',
      '</div>'
      ;

      if( empty( $presets_list) ) {
        echo '<div class="awf-templates-empty-notice">' . esc_html__( 'There are no preset templates yet. Please use the "Create" and "Import" buttons to add templates.', 'annasta-filters' ) . '</div>';

      } else {
        echo '<div class="awf-preset-templates-list-container">', $presets_list, '</div>';
      }

      echo
        '<div class="awf-templates-subheader">',
        '<h3>', esc_html__( 'Filter templates', 'annasta-filters' ), '</h3>',
        '<div class="awf-settings-tb-row-buttons-container">',
        '<button type="button" title="', esc_attr__( 'Create filter template', 'annasta-filters' ), '" class="button button-secondary awf-fa-icon-text-btn awf-popup-create-filter-template-btn">' . esc_html__( 'Create', 'annasta-filters' ) . '</button>',
        '<button type="button" title="' . esc_attr__( 'Import template from JSON file', 'annasta-filters' ) . '" class="button button-secondary awf-fa-icon-text-btn awf-import-filter-template-btn">' . esc_html__( 'Import', 'annasta-filters' ) . '</button>',
        '<input type="file" id="awf-import-json-filter-template" accept=".json" style="display:none;" />',
        '</div>',
        '</div>',

        '<div class="awf-settings-tb-header-row">',
        $filters_select_html,
        '</div>'
      ;

      echo '<div class="awf-filter-templates-list-container">' . $filters_list . '</div>';
    }

    private function display_create_preset_template_popup() {
      $presets_names = $this->get_presets_names();

      $html = '<div class="awf-overlay-popup awf-create-preset-template-popup">' .
      '<div title="' . esc_attr__( 'Close', 'annasta-filters' ) . '" class="awf-fa-icon awf-close-overlay-popup-btn"></div>' .

      '<div class="awf-overlay-popup-row awf-overlay-popup-header-row">' .
      esc_html__( 'Create template', 'annasta-filters' ) .
      '</div>'
      ;

      if( empty( $presets_names ) ) {
        $html .= '<div class="awf-overlay-popup-row awf-overlay-popup-subheading">' .
        '<label>' . esc_html__( 'No filter presets have been found!', 'annasta-filters' ) . '</label>' .
        '</div>'
        ;

      } else {
        $preset_select = A_W_F::$admin->build_select_html(
          array(
            'id' => 'awf-create-preset-template-presets-select',
            'options' => $presets_names,
          )
        );

          $html .= '<div class="awf-overlay-popup-row awf-create-preset-template-select-container">' .
          '<label for="awf-create-preset-template-presets-select">' . esc_html__( 'Preset', 'annasta-filters' ) . '</label>' . $preset_select .
          '</div>'
          ;

          $html .=
          '<div class="awf-overlay-popup-row awf-create-preset-template-btn-container">' .
          '<input class="awf-new-preset-template-name" type="text" class="" value="" placeholder="' . esc_attr__( 'Template name', 'annasta-filters' ) . '"></input>' .
          '<button type="button" title="' . esc_attr__( 'Save selected preset as template', 'annasta-filters' ) . '" class="button button-secondary awf-create-preset-template-btn">' . esc_html__( 'Save as template', 'annasta-filters' ) . '</button>' .
          '</div>'
          ;
      }

      $html .= '</div>';

      echo $html;
    }
    
    private function create_preset_template( $preset_id, $template_name ) {

      if( ! isset(  A_W_F::$presets[$preset_id] ) ) { return false; }

      $templates = $this->get_templates();

      $template = array(
        'template_name' => $template_name,
        'awf_version' => A_W_F_VERSION,
        'settings' => array(),
        'filters_templates' => array(),
        'associations' => A_W_F::$presets[$preset_id]['associations'],
      );

      $all_options = wp_load_alloptions();
      $prefix = 'awf_preset_' . $preset_id . '_';
      $prefix_length = strlen( $prefix );

      foreach( $all_options as $option => $value ) {
        if( 0 !== strpos( $option, $prefix ) ) { continue; }
        $clean_option = substr( $option, $prefix_length );
        $template['settings'][$clean_option] = $value;
      }

      if( empty( $template['template_name'] ) && isset( $template['settings']['name'] ) ) {
        $template['template_name'] = sprintf( __( '%1$s template', 'annasta-filters' ), $template['settings']['name'] );
      }

      foreach( A_W_F::$presets[$preset_id]['filters'] as $filter_id => $position ) {
        $filter = new A_W_F_filter( $preset_id, $filter_id );
  
        $filter_template = array(
          'template_name' => sprintf( __( '%1$s template', 'annasta-filters' ), $template['template_name'] . ' > ' . $this->get_default_filter_label( $filter->module, $filter->settings ) ),
          'linked_preset' => $preset_id,
          'awf_version' => A_W_F_VERSION,
          'module' => $filter->module,
          'name' => $filter->name,
          'settings' => $filter->settings,
        );
  
        $template['filters_templates'][$position] = $filter_template;
      }

      $templates['presets'][] = $template;
      update_option( 'awf_templates', $templates );

      end( $templates['presets'] );
      return intval( key( $templates['presets'] ) );
    }

    private function display_rename_preset_template_popup( $template_id ) {
      $all_templates = $this->get_templates();

      $html = '<div class="awf-overlay-popup awf-popup-rename-preset-template">' .
      '<div title="' . esc_attr__( 'Close', 'annasta-filters' ) . '" class="awf-fa-icon awf-close-overlay-popup-btn"></div>' .

      '<div class="awf-overlay-popup-row awf-overlay-popup-header-row">' .
      esc_html__( 'Rename template', 'annasta-filters' ) .
      '</div>'
      ;

      if( isset( $all_templates['presets'][$template_id] ) ) {
        $html .=
        '<div class="awf-overlay-popup-row">' .
        '<input class="awf-rename-preset-template" type="text" value="' . $all_templates['presets'][$template_id]['template_name'] . '"></input>' .
        '<button type="button" title="' . esc_attr__( 'Rename template', 'annasta-filters' ) . '" class="button button-secondary awf-rename-preset-template-btn" data-preset-template-id="' . $template_id . '">' . esc_html__( 'Rename', 'annasta-filters' ) . '</button>' .
        '</div>'
        ;
      }

      $html .= '</div>';

      echo $html;
    }

    private function rename_preset_template( $template_id, $template_name ) {
      $templates = $this->get_templates();

      if( isset( $templates['presets'][$template_id] ) ) {
        $templates['presets'][$template_id]['template_name'] = $template_name;
      }

      update_option( 'awf_templates', $templates );
    }
    
    private function delete_preset_template( $template_id ) {
      $templates = $this->get_templates();

      if( isset( $templates['presets'][$template_id] ) ) {
        $templates['presets'] = array_diff_key( $templates['presets'], array( $template_id => array() ) );
      }

      update_option( 'awf_templates', $templates );
    }

    private function export_preset_template( $template_id ) {
      $templates = $this->get_templates();

      if( isset( $templates['presets'][$template_id] ) ) {
        $json_template = json_encode( $templates['presets'][$template_id] );
        $file_name = time() . '.json';
        $pretty_file_name = 'annasta-' . sanitize_title( $templates['presets'][$template_id]['template_name'], $template_id ) . '-preset-template.json';

        $awf_uploads_folder = trailingslashit( wp_upload_dir()['basedir'] ) . 'annasta-filters/templates';
        $path = trailingslashit( $awf_uploads_folder ) . $file_name;
        $url = trailingslashit( trailingslashit( wp_upload_dir()['baseurl'] ) . 'annasta-filters/templates' ) . $file_name;

        if( wp_mkdir_p( $awf_uploads_folder ) ) {
          require_once( ABSPATH . 'wp-admin/includes/file.php' );
          $old_files = list_files( $awf_uploads_folder, 1 );
          if( $old_files ) {
            foreach( $old_files as $file ) {
              if ( is_file( $file ) ) {
                $difference = (int) time() - (int) basename( $file, '.json' );
                if( $difference > 90000 ) {
                  unlink( $file );
                }
              }
            }
          }
          
          file_put_contents( $path, $json_template );

          $response = '<div class="awf-overlay-popup-row awf-info-notice">' . sprintf( wp_kses( __( 'A JSON file for the "%3$s" preset template has been created. Please <a href="%1$s" class="awf-preset-template-download-link" download="%2$s">click here</a> if the download hasn\'t started automatically.', 'annasta-filters' ), array( 'a' => array( 'href' => array(), 'class' => array(), 'download' => array() ) ) ), esc_url( $url ), $pretty_file_name, $templates['presets'][$template_id]['template_name'] ) . '</div>';

          echo $response;

        } else {
          echo '<div class="awf-overlay-popup-row awf-info-notice">Error exporting file</div>';
        }
      }
    }
        
    private function import_preset_template( $imported_template ) {

      if( ! is_array( $imported_template )
        || ! isset( $imported_template['template_name'] )
        || ! isset( $imported_template['awf_version'] )
        || ! isset( $imported_template['settings'] )
        || ! isset( $imported_template['filters_templates'] )
        || ! isset( $imported_template['associations'] )
      ) {
        return json_encode( array( 'notices' => esc_attr__( 'Import operation aborted: provided file is of invalid format.', 'annasta-filters' ) ) );
      }

      $notices = array();
      $template = array();

      $template['template_name'] = sanitize_text_field( wp_unslash( $imported_template['template_name'] ) );
      $template['awf_version'] = sanitize_text_field( $imported_template['awf_version'] );
      if( $template['awf_version'] !== A_W_F_VERSION ) {
        $notices[] = sprintf( __( 'Warning: The newly imported template was created in the version %2$s of annasta Woocommerce Product Filters. To ensure the imported template compatibility with your filters, it is recommended that it were created in the same version with the one that is currently installed. Current plugin version number: %1$s.', 'annasta-filters' ), A_W_F_VERSION, $template['awf_version'] );
      }

      foreach( $imported_template['settings'] as $setting_id => $setting ) {
        if( is_string( $setting ) ) {
          $template['settings'][sanitize_key( $setting_id )] = $this->sanitize_imported_template_string( $setting_id, $setting );

        } elseif( is_array( $setting ) ) {
          $template['settings'][sanitize_key( $setting_id )] = $this->sanitize_imported_template_array( $setting );

        } elseif( is_bool( $setting ) || is_null( $setting ) || is_numeric( $setting ) ) {
          $template['settings'][sanitize_key( $setting_id )] = $setting;
        }
      }

      $position = 0;

      foreach( $imported_template['filters_templates'] as $p => $filter_template ) {
        $new_template_data = $this->build_imported_filter_template( $filter_template );
        if( ! empty( $new_template_data['template'] ) ) {
          $template['filters_templates'][$position++] = $new_template_data['template'];
        }
      }

      foreach( $imported_template['associations'] as $association_id => $association ) {
        $template['associations'][intval( $association_id )] = sanitize_textarea_field( $association );
      }
      
      $templates = $this->get_templates();
      $templates['presets'][] = $template;
      update_option( 'awf_templates', $templates );

      array_unshift( $notices, sprintf( __( 'New template "%1$s" has been successfuly imported.', 'annasta-filters' ), $template['template_name'] ) );

      $response = array( 'notices' => implode( "\r\n\r\n", $notices ) );

      return json_encode( $response );
    }

    private function display_apply_preset_template_popup( $preset_id ) {

      $preset_templates = $cloneable_presets = array();
      $all_templates = $this->get_templates();

      $html = '<div class="awf-overlay-popup awf-preset-templates" data-preset-id="' . $preset_id . '">' .
      '<div title="' . esc_attr__( 'Close', 'annasta-filters' ) . '" class="awf-fa-icon awf-close-overlay-popup-btn"></div>' .

      '<div class="awf-overlay-popup-row awf-overlay-popup-header-row awf-valign">' .
      '<div>' . esc_html__( 'Import settings from another preset or template', 'annasta-filters' ) . '</div>' .
      '<div class="awf-overlay-popup-subcaption">' . esc_html__( 'Applying template or transferring settings from another preset will replace all the settings and filters of the current preset.', 'annasta-filters' ) . '</div>' .
      '</div>'
      ;

      if( count( A_W_F::$presets ) > 1 ) {
        foreach( A_W_F::$presets as $pid => $preset ) {
          if( intval( $pid ) !== $preset_id ) {
            $cloneable_presets[$pid] = get_option( 'awf_preset_' . $pid . '_name', 'Unnamed Preset' );
          }
        }
      }

      foreach( $all_templates['presets'] as $template_id => $template ) {
        $preset_templates[$template_id] = $template['template_name'];
      }

      if( empty( $cloneable_presets ) ) {

        if( empty( $preset_templates ) ) {
          $html .= '<div class="awf-overlay-popup-row awf-info-notice">';
          $html .= esc_html__( 'No additional presets or templates have been found on this installation. Please add more presets or go to annasta Filters > Templates admin section to import templates.', 'annasta-filters' );
          $html .= '</div>';

        }

      } else {

        $presets_select = array(
          'id' => 'awf-apply-preset-clone-select',
          'options' => $cloneable_presets
        );

        $html .= '<div class="awf-overlay-popup-row awf-apply-preset-clone-container awf-valign">';
        $html .= '<label for="awf-apply-preset-clone-select">' . esc_html__( 'From another preset', 'annasta-filters' ) . '</label>';
        $html .= $this->build_select_html( $presets_select );
        $html .= '<button type="button" title="' . esc_attr__( 'Import into the current preset', 'annasta-filters' ) . '" class="button button-secondary awf-apply-preset-clone-btn">' . sprintf( esc_html__( 'Import into %1$s', 'annasta-filters' ), '<strong>' . get_option( 'awf_preset_' . $preset_id . '_name', '' ) ) . '</strong>' . '</button>';
        $html .= '</div>';
      }


      if( ! empty( $preset_templates ) ) {
        $templates_select = array(
          'id' => 'awf-apply-preset-template-select',
          'options' => $preset_templates
        );

        $html .= '<div class="awf-overlay-popup-row awf-apply-preset-template-container">';
        $html .= '<label for="awf-apply-preset-template-select">' . esc_html__( 'From template', 'annasta-filters' ) . '</label>';
        $html .= $this->build_select_html( $templates_select );
        $html .= '<button type="button" title="' . esc_attr__( 'Apply to current preset', 'annasta-filters' ) . '" class="button button-secondary awf-apply-preset-template-btn">' . sprintf( esc_html__( 'Apply to %1$s', 'annasta-filters' ), '<strong>' . get_option( 'awf_preset_' . $preset_id . '_name', '' ) ) . '</strong>' . '</button>';
        $html .= '</div>';
      }

      echo $html;
    }

    private function apply_preset_clone( $preset_id, $parent_id ) {

      $template_id = $this->create_preset_template( $parent_id, 'Preset #' . $parent_id . ' Clone' );
      $name = get_option( 'awf_preset_' . $preset_id . '_name', '' );

      if( false === $template_id ) { die(); }

      $this->apply_preset_template( $preset_id, $template_id );
      update_option( 'awf_preset_' . $preset_id . '_name', $name );
      $this->delete_preset_template( intval( $template_id ) );
    }

    private function apply_preset_template( $preset_id, $template_id ) {
      $templates = $this->get_templates();

      if( ! isset( A_W_F::$presets[$preset_id] ) || ! isset( $templates['presets'][$template_id] ) ) { return; }

      $prefix = 'awf_preset_' . $preset_id . '_';

      foreach( $templates['presets'][$template_id]['settings'] as $setting => $value ) {
        update_option( $prefix . sanitize_key( $setting ), $value );
      }

      A_W_F::$presets[$preset_id]['associations'] = array();
      $all_associations = $this->get_all_associations();

      foreach( $templates['presets'][$template_id]['associations'] as $association_id => $association ) {
        if( isset( $all_associations[$association] ) ) {
          A_W_F::$presets[$preset_id]['associations'][] = $association;
        }
      }
      update_option( 'awf_presets', A_W_F::$presets );
      
      foreach( A_W_F::$presets[$preset_id]['filters'] as $filter_id => $position ) {
        $filter = new A_W_F_filter( $preset_id, $filter_id );
        $this->delete_filter( $filter );
      }

      $all_filters = A_W_F::$admin->get_all_filters();

      foreach( $templates['presets'][$template_id]['filters_templates'] as $position => $template ) {
        if( isset( $all_filters[$template['name']] ) ) {
          $filter = $this->add_filter( $preset_id, $template['name'] );
          if( $filter instanceof A_W_F_filter ) {
            $this->apply_filter_template( $filter, $template );
          }
        }
      }

      /* A_W_F::build_query_vars() call is needed for meta filters if
       * the preset form submission is not forced after template application
       */

      $this->generate_styles_css();

    }

    private function display_create_filter_template_popup( $preset_id = false ) {
      $presets_names = $this->get_presets_names();
      $all_filters = A_W_F::$admin->get_all_filters();
      $preset_filters = array();

      if( false === $preset_id ) {
        $preset_ids = array_keys( A_W_F::$presets );
        $preset_id = reset( $preset_ids );
      }

      if( isset( A_W_F::$presets[$preset_id] ) ) {
        foreach( A_W_F::$presets[$preset_id]['filters'] as $filter_id => $position ) {
          $filter = new A_W_F_filter( $preset_id, $filter_id );
          $preset_filters[$filter_id] = $filter->settings['title'];

          if( empty( $preset_filters[$filter_id] ) && isset( $all_filters[$filter->name] ) ) {
            $preset_filters[$filter_id] = $all_filters[$filter->name];
          }
        }
      }

      $html = '<div class="awf-overlay-popup awf-create-filter-template-popup">' .
      '<div title="' . esc_attr__( 'Close', 'annasta-filters' ) . '" class="awf-fa-icon awf-close-overlay-popup-btn"></div>' .

      '<div class="awf-overlay-popup-row awf-overlay-popup-header-row">' .
      esc_html__( 'Create template', 'annasta-filters' ) .
      '</div>'
      ;

      if( empty( $presets_names ) ) {
        $html .= '<div class="awf-overlay-popup-row awf-overlay-popup-subheading">' .
        '<label>' . esc_html__( 'No filter presets have been found!', 'annasta-filters' ) . '</label>' .
        '</div>'
        ;

      } else {
        $preset_select = A_W_F::$admin->build_select_html(
          array(
            'id' => 'awf-create-filter-template-presets-select',
            'options' => $presets_names,
            'selected' => false === $preset_id ? '' : $preset_id
          )
        );

          $html .= '<div class="awf-overlay-popup-row awf-create-filter-template-select-container">' .
          '<label for="awf-create-filter-template-presets-select">' . esc_html__( 'Preset', 'annasta-filters' ) . '</label>' . $preset_select .
          '</div>'
          ;

        if( empty( $preset_filters ) ) {
          $html .= '<div class="awf-overlay-popup-row awf-overlay-popup-subheading">' .
          '<label>' . esc_html__( 'Preset has no filters', 'annasta-filters' ) . '</label>' .
          '</div>'
          ;

        } else {
          $filters_select = A_W_F::$admin->build_select_html(
            array(
              'id' => 'awf-create-filter-template-filters-select',
              'options' => $preset_filters,
            )
          );

          $html .= '<div class="awf-overlay-popup-row awf-create-filter-template-select-container">' .
          '<label for="awf-create-filter-template-filters-select">' . esc_html__( 'Filter', 'annasta-filters' ) . '</label>' . $filters_select .
          '</div>'
          ;

          $html .=
          '<div class="awf-overlay-popup-row awf-create-filter-template-btn-container">' .
          '<input class="awf-new-filter-template-name" type="text" class="" value="" placeholder="' . esc_attr__( 'Template name', 'annasta-filters' ) . '"></input>' .
          '<button type="button" title="' . esc_attr__( 'Save selected filter as template', 'annasta-filters' ) . '" class="button button-secondary awf-create-filter-template-btn">' . esc_html__( 'Save as template', 'annasta-filters' ) . '</button>' .
          '</div>'
          ;
        }        
      }

      $html .= '</div>';

      echo $html;
    }

    private function create_filter_template( $filter, $template_name = '' ) {

      if( empty( $template_name ) ) {
        $template_name = sprintf( __( '%1$s template', 'annasta-filters' ), get_option( 'awf_preset_' . $filter->preset_id . '_name', '' ) . ' > ' . $this->get_default_filter_label( $filter->module, $filter->settings ) );
      }

      $template = array(
        'template_name' => $template_name,
        'awf_version' => A_W_F_VERSION,
        'module' => $filter->module,
        'name' => $filter->name,
        'settings' => $filter->settings,
      );

      $templates = $this->get_templates();
      $templates['filters'][] = $template;

      update_option( 'awf_templates', $templates );

      end( $templates['filters'] );
      return intval( key( $templates['filters'] ) );
    }

    private function display_rename_filter_template_popup( $template_id ) {
      $all_templates = $this->get_templates();

      $html = '<div class="awf-overlay-popup awf-popup-rename-filter-template">' .
      '<div title="' . esc_attr__( 'Close', 'annasta-filters' ) . '" class="awf-fa-icon awf-close-overlay-popup-btn"></div>' .

      '<div class="awf-overlay-popup-row awf-overlay-popup-header-row">' .
      esc_html__( 'Rename template', 'annasta-filters' ) .
      '</div>'
      ;

      if( isset( $all_templates['filters'][$template_id] ) ) {
        $html .=
        '<div class="awf-overlay-popup-row">' .
        '<input class="awf-rename-filter-template" type="text" value="' . $all_templates['filters'][$template_id]['template_name'] . '"></input>' .
        '<button type="button" title="' . esc_attr__( 'Rename template', 'annasta-filters' ) . '" class="button button-secondary awf-rename-filter-template-btn" data-filter-template-id="' . $template_id . '">' . esc_html__( 'Rename', 'annasta-filters' ) . '</button>' .
        '</div>'
        ;
      }

      $html .= '</div>';

      echo $html;
    }

    private function rename_filter_template( $template_id, $template_name ) {
      $templates = $this->get_templates();

      if( isset( $templates['filters'][$template_id] ) ) {
        $templates['filters'][$template_id]['template_name'] = $template_name;
      }

      update_option( 'awf_templates', $templates );
    }
    
    private function delete_filter_template( $template_id ) {
      $templates = $this->get_templates();

      if( isset( $templates['filters'][$template_id] ) ) {
        $templates['filters'] = array_diff_key( $templates['filters'], array( $template_id => array() ) );
      }

      update_option( 'awf_templates', $templates );
    }
        
    private function export_filter_template( $template_id ) {
      $templates = $this->get_templates();

      if( isset( $templates['filters'][$template_id] ) ) {
        $json_template = json_encode( $templates['filters'][$template_id] );
        $file_name = time() . '.json';
        $pretty_file_name = 'annasta-filters-template-' . $template_id . '.json';

        $awf_uploads_folder = trailingslashit( wp_upload_dir()['basedir'] ) . 'annasta-filters/templates';
        $path = trailingslashit( $awf_uploads_folder ) . $file_name;
        $url = trailingslashit( trailingslashit( wp_upload_dir()['baseurl'] ) . 'annasta-filters/templates' ) . $file_name;

        if( wp_mkdir_p( $awf_uploads_folder ) ) {
          require_once( ABSPATH . 'wp-admin/includes/file.php' );
          $old_files = list_files( $awf_uploads_folder, 1 );
          if( $old_files ) {
            foreach( $old_files as $file ) {
              if ( is_file( $file ) ) {
                $difference = (int) time() - (int) basename( $file, '.json' );
                if( $difference > 90000 ) {
                  unlink( $file );
                }
              }
            }
          }
          
          file_put_contents( $path, $json_template );

          $response = '<div class="awf-overlay-popup-row awf-info-notice">' . sprintf( wp_kses( __( 'A JSON file for the "%3$s" filter template has been created. Please <a href="%1$s" class="awf-filter-template-download-link" download="%2$s">click here</a> if the download hasn\'t started automatically.', 'annasta-filters' ), array( 'a' => array( 'href' => array(), 'class' => array(), 'download' => array() ) ) ), esc_url( $url ), $pretty_file_name, $templates['filters'][$template_id]['template_name'] ) . '</div>';

          echo $response;

        } else {
          echo '<div class="awf-overlay-popup-row awf-info-notice">Error exporting file</div>';
        }
      }
    }
        
    private function import_filter_template( $imported_template ) {
      $response = array();
      $template_data = $this->build_imported_filter_template( $imported_template );

      if( ! empty( $template_data['template'] ) ) {
        $templates = $this->get_templates();
        $templates['filters'][] = $template_data['template'];
        update_option( 'awf_templates', $templates );

        $response['filter_name'] = $template_data['template']['name'];
        array_unshift( $template_data['notices'], sprintf( __( 'New template "%1$s" has been successfuly imported.', 'annasta-filters' ), $template_data['template']['template_name'] ) );
      }

      $response['notices'] = implode( "\r\n\r\n", $template_data['notices'] );

      echo( json_encode( $response ) );
    }
    
    private function build_imported_filter_template( $imported_template ) {

      if( ! is_array( $imported_template )
      || ! isset( $imported_template['module'] )
      || ! isset( $imported_template['name'] )
      || ! isset( $imported_template['awf_version'] )
      || ! isset( $imported_template['template_name'] )
      || ! isset( $imported_template['settings'] )
      ) {
        return array( 'notices' => array( esc_attr__( 'Import operation aborted: provided file is of invalid format.', 'annasta-filters' ) ) );
      }

      $response = array( 'notices' => array(), 'template' => array() );
      $response['template']['module'] = sanitize_text_field( $imported_template['module'] );
      $response['template']['name'] = sanitize_text_field( $imported_template['name'] );
      $default_data = $this->build_new_filter_data( $response['template']['name'] );

      if( empty( $default_data ) || $response['template']['module'] !== $default_data['module'] ) {
        $response['notices'][] = __( 'Import operation aborted: invalid template parameters.', 'annasta-filters' );
        $response['template'] = array();

      } else {
        $response['template']['template_name'] = sanitize_text_field( wp_unslash( $imported_template['template_name'] ) );
        $response['template']['awf_version'] = sanitize_text_field( $imported_template['awf_version'] );
        if( $response['template']['awf_version'] !== A_W_F_VERSION ) {
          $response['notices'][] = sprintf( __( 'Warning: The newly imported template was created in the version %2$s of annasta Woocommerce Product Filters. To ensure the imported template compatibility with your filters, it is recommended that it were created in the same version with the one that is currently installed. Current plugin version number: %1$s.', 'annasta-filters' ), A_W_F_VERSION, $response['template']['awf_version'] );
        }
        if( isset( $imported_template['linked_preset'] ) ) { $response['template']['linked_preset'] = intval( $imported_template['linked_preset'] ); }

        foreach( $imported_template['settings'] as $setting_id => $setting ) {
          $sanitized_id = is_numeric( $setting_id ) ? $setting_id : sanitize_text_field( $setting_id );

          if( is_string( $setting ) ) {
            if( empty( $setting ) && in_array( $setting_id, array( 'type', 'style' ), true ) ) {
              $sanitized_array[$sanitized_id] = null;
            } else {
              $response['template']['settings'][$sanitized_id] = $this->sanitize_imported_template_string( $sanitized_id, $setting );
            }

          } elseif( is_array( $setting ) ) {
            $response['template']['settings'][$sanitized_id] = $this->sanitize_imported_template_array( $setting );

          } elseif( is_bool( $setting ) || is_null( $setting ) || is_numeric( $setting ) ) {
            $response['template']['settings'][$sanitized_id] = $setting;
          }
        }
      }

      return $response;
    }
    
    private function sanitize_imported_template_array( $array ) {
      $sanitized_array = array();

      foreach( $array as $setting_id => $setting ) {
        $sanitized_id = is_numeric( $setting_id ) ? $setting_id : sanitize_text_field( $setting_id );

        if( is_string( $setting ) ) {
          $sanitized_array[$sanitized_id] = $this->sanitize_imported_template_string( $sanitized_id, $setting );
          
        } elseif( is_array( $setting ) ) {
          $sanitized_array[$sanitized_id] = $this->sanitize_imported_template_array( $setting );

        } elseif( is_bool( $setting ) || is_null( $setting ) || is_numeric( $setting ) ) {
          $sanitized_array[$sanitized_id] = $setting;
        }
      }

      return $sanitized_array;
    }
    
    private function sanitize_imported_template_string( $setting_id, $value ) {
      if( in_array( $setting_id, array( 'hierarchical_level', 'height_limit', 'autocomplete_after', 'autocomplete_results_count', 'autocomplete_height_limit' ), true ) ) {
        $value = (int) $value;

      } else {
        switch( $value ) {
          case 'false':
            $value = false;
            break;
          case 'true':
            $value = true;
            break;
          default:
            $value = sanitize_textarea_field( wp_unslash( $value ) );
            break;
        }
      }

      return $value;
    }

    private function display_apply_filter_template_popup( $filter, $notice = '' ) {
      $filter_templates = $cloneable_filters = array();
      $all_templates = $this->get_templates();
      $filter_label = get_option( 'awf_preset_' . $filter->preset_id . '_name', '' ) . ' > ' . $filter->settings['title'];

      $html = '<div class="awf-overlay-popup awf-filter-templates" data-filter-id="' . $filter->id . '">' .
      '<div title="' . esc_attr__( 'Close', 'annasta-filters' ) . '" class="awf-fa-icon awf-close-overlay-popup-btn"></div>' .

      '<div class="awf-overlay-popup-row awf-overlay-popup-header-row">' .
      esc_html__( 'Import settings from another filter or template', 'annasta-filters' ) .
      '</div>'
      ;

      if( ! empty( $notice ) ) {
        $html .= '<div class="awf-overlay-popup-row awf-info-notice">' . esc_html( $notice ) . '</div>';
      }

      if( count( A_W_F::$presets ) > 1 ) {
        foreach( A_W_F::$presets as $pid => $preset ) {

          if( intval( $pid ) === $filter->preset_id ) { continue; }

          foreach( $preset['filters'] as $fid => $position ) {
            $prefix = A_W_F_filter::get_prefix( $pid, $fid );
            if( $filter->name === get_option( $prefix . 'name', '' ) ) {
              $f = new A_W_F_filter( $pid, $fid );
              $cloneable_filters[$prefix] = get_option( 'awf_preset_' . $pid . '_name', '' ) . ' > ' . A_W_F::$admin->get_default_filter_label( $f->module, $f->settings );
            }
          }
        }
      }

      foreach( $all_templates['filters'] as $template_id => $template ) {
        if( isset( $template['module'] ) && $filter->module === $template['module'] && $filter->name === $template['name'] ) {

          if( 'meta' === $filter->module ) {
            if( empty( $filter->settings['meta_name'] ) || $filter->settings['meta_name'] !== $template['settings']['meta_name']) {
              continue;
            }
          }

          $filter_templates[$template_id] = $template['template_name'];
        }
      }

      if( empty( $cloneable_filters ) ) {

        if( empty( $filter_templates ) ) {
          $html .= '<div class="awf-overlay-popup-row awf-info-notice">';
          $html .= esc_html__( 'No matching filters or templates were found on this installation. Please create the same kind of filter in another preset or import a matching template in the annasta Filters > Templates section of your admin.', 'annasta-filters' );
          $html .= '</div>';
        }

      } else {
          $filters_select = array(
          'id' => 'awf-apply-filter-clone-select',
          'options' => $cloneable_filters
          );

          $html .= '<div class="awf-overlay-popup-row awf-apply-filter-clone-container awf-valign">';
          $html .= '<label for="awf-apply-filter-clone-select">' . esc_html__( 'From another filter', 'annasta-filters' ) . '</label>';
          $html .= $this->build_select_html( $filters_select );
          $html .= '<button type="button" title="' . esc_attr__( 'Import into the current filter', 'annasta-filters' ) . '" class="button button-secondary awf-apply-filter-clone-btn"><strong>' . sprintf( esc_html__( 'Import into %1$s', 'annasta-filters' ), $filter_label ) . '</strong>' . '</button>';
          $html .= '</div>';          
      }

      if( ! empty( $filter_templates ) ) {
        $templates_select = array(
          'id' => 'awf-apply-filter-template-select',
          'options' => $filter_templates
        );

        $html .= '<div class="awf-overlay-popup-row awf-apply-filter-template-container">';
        $html .= '<label for="awf-apply-filter-template-select">' . esc_html__( 'Template', 'annasta-filters' ) . '</label>';
        $html .= $this->build_select_html( $templates_select );
        $html .= '<button type="button" title="' . esc_attr__( 'Apply to current filter', 'annasta-filters' ) . '" class="button button-secondary awf-apply-filter-template-btn">' . sprintf( esc_html__( 'Apply to %1$s', 'annasta-filters' ), '<strong>' . $filter_label ) . '</strong>' . '</button>';
        $html .= '</div>';
      }

      echo $html;
    }

    private function apply_filter_template( $filter, $template ) {
      if( $filter->name !== $template['name'] ) { return; }

      $default_data = $this->build_new_filter_data( $template['name'] );

      if( ! empty( $default_data ) ) {
        $settings = $this->get_module_defaults( $default_data );

        foreach( $settings as $setting => $value ) {
          if( isset( $template['settings'][$setting] ) ) {
            $settings[$setting] = $template['settings'][$setting];
          }
        }

        $filter->settings = $settings;
        update_option( $filter->prefix . 'settings', $filter->settings );
      }
    }
    
    private function apply_filter_clone( $filter ) {
      
      if( empty( $filter->settings ) || ! isset( A_W_F::$presets[$filter->preset_id]['filters'][$filter->id] ) ) { die(); }

      $title = $filter->settings['title'];

      $d = sanitize_key( $_POST['awf_clone_filter'] );
      $d = rtrim( $d, '_' );
      $d = substr( $d, strlen( 'awf_filter_' ) );
      $parent_data = explode( '-', $d );

      if( 2 !== count( $parent_data ) ) { die(); }

      $parent = new A_W_F_filter( intval( $parent_data[0] ), intval( $parent_data[1] ) );
      if( empty( $parent->settings) || ! isset( A_W_F::$presets[$parent->preset_id]['filters'][$parent->id] ) ) { die(); }

      $template_id = $this->create_filter_template( $parent );
      $templates = $this->get_templates();

      $this->apply_filter_template( $filter, $templates['filters'][$template_id] );

      $settings = get_option( $filter->prefix . 'settings', array() );
      $settings['title'] = $title;
      update_option( $filter->prefix . 'settings', $settings );

      $this->delete_filter_template( $template_id );
    }

    private function get_templates() {
      return get_option(
        'awf_templates',
        array(
          'presets' => array(),
          'filters' => array(),
        )
      );
    }
    
    private function check_permissions() {
      $awf_nonce = $_GET['awf-nonce'];
      $nonce_check = 'awf-' . sanitize_key( $_GET['awf-action'] );

      if( ! current_user_can( 'manage_woocommerce' ) || ! wp_verify_nonce( $awf_nonce, $nonce_check ) ) {
        $this->display_admin_notice( __( 'You don\'t have sufficient permissions to perform this action. ', 'annasta-filters' ) );
        die();
      }
    }

  }
}

?>